/*!
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.3
 *
 * @date        2022-09-20
 *
 * @attention
 *
 *  Copyright (C) 2020-2022 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes */
#include "stdio.h"
#include "apm32f0xx_gpio.h"
#include "apm32f0xx_misc.h"
#include "apm32f0xx_eint.h"
#include "apm32f0xx_rtc.h"
#include "apm32f0xx_pmu.h"
#include "Board.h"

/** @addtogroup Examples
  @{
  */

/** @addtogroup RTC_Stamp
  @{
  */

/** @defgroup RTC_Stamp_Macros Macros
  @{
*/

/* printf function config to USART1 */
#define DEBUG_USART  USART1

/**@} end of group RTC_Stamp_Macros */

/** @defgroup RTC_Stamp_Enumerations Enumerations
  @{
*/

/**@} end of group RTC_Stamp_Enumerations */

/** @defgroup RTC_Stamp_Structures Structures
  @{
*/

/**@} end of group RTC_Stamp_Structures */

/** @defgroup RTC_Stamp_Variables Variables
  @{
*/

/**@} end of group RTC_Stamp_Variables */

/** @defgroup RTC_Stamp_Functions Functions
  @{
  */

/* Delay */
void Delay(void);
/* RTC Init */
void RTC_Init(void);
/* RTC Eint */
void RTC_EintConfig(void);

/*!
 * @brief       Main program
 *
 * @param       None
 *
 * @retval      None
 *
 * @note
 */
int main(void)
{
    uint8_t Presec;
    RTC_DATE_T DateStruct;
    RTC_TIME_T TimeStruct;

    APM_MINI_LEDInit(LED2);
    APM_MINI_LEDInit(LED3);
    APM_MINI_COMInit(COM1);

    /* RTC Reset */
    RTC_Init();
    RTC_Reset();
    RTC_Init();

    /* RTC Enable*/
    RTC_EnableInit();

    /* Init Date 20/8/22/6 */
    DateStruct.year =  20;
    DateStruct.month = 8;
    DateStruct.date =  22;
    DateStruct.weekday = 6;
    RTC_ConfigDate(RTC_FORMAT_BIN, &DateStruct);
    Delay();

    /* Init Time 23:59:50 */
    TimeStruct.H12 = 12;
    TimeStruct.hours = 23;
    TimeStruct.minutes = 59;
    TimeStruct.seconds = 50;
    RTC_ConfigTime(RTC_FORMAT_BIN, &TimeStruct);
    Delay();

    printf("\r\nConnect PC13 in the air to require a TAMPER1\r\n");
    printf("\r\nConnect PA0  in the air to require a TAMPER2\r\n");
    /* RTC Eint Line Init */
    RTC_EintConfig();

    /* Config Tamper */
    RTC_ConfigTamperTrigger(RTC_TAMPER_1, RTC_TAMPER_TRIGGER_RISINGEDGE);
    RTC_EnableTamper(RTC_TAMPER_1);

    RTC_ConfigTamperTrigger(RTC_TAMPER_2, RTC_TAMPER_TRIGGER_RISINGEDGE);
    RTC_EnableTamper(RTC_TAMPER_2);

    /* Config Time Stamp */
    RTC_EnableTimeStamp(RTC_TIME_STAMPEDGE_RISING);

    /* Enable TimeStamp Interrupt */
    RTC_EnableInterrupt(RTC_INT_FLAG_TAMP1);
    RTC_EnableInterrupt(RTC_INT_FLAG_TAMP2);

    /* Enable TimeStamp Interrupt */
    RTC_EnableInterrupt(RTC_INT_FLAG_TS);
    NVIC_EnableIRQRequest(RTC_IRQn, 2);

    RTC_DisableInit();

    for (;;)
    {
        Presec = TimeStruct.seconds;

        RTC_ReadTime(RTC_FORMAT_BIN, &TimeStruct);

        /* Read Date and Time per second */
        if (Presec != TimeStruct.seconds)
        {
            RTC_ReadDate(RTC_FORMAT_BIN, &DateStruct);
            printf(" \r\n date = 20%d / %d / %d / %d \r\n",
                   DateStruct.year, DateStruct.month, DateStruct.date, DateStruct.weekday);

            printf(" \r\n time : %d - %d - %d \r\n",
                   (TimeStruct.hours), (TimeStruct.minutes), (TimeStruct.seconds));
        }
    }
}

/*!
 * @brief       RTC Isr
 *
 * @param       None
 *
 * @retval      None
 *
 * @note        This function need to put into RTC_IRQHandler in apm32f0xx_int.h
 */

void RTC_Isr(void)
{
    RTC_DATE_T dateStruct;
    RTC_TIME_T timeStruct;

    if (RTC_ReadIntFlag(RTC_INT_FLAG_TAMP1) == SET)
    {
        printf("RTC_INT_TAMP1\r\n");

        APM_MINI_LEDOn(LED2);

        /* Clear RTC TAMP1 flag */
        RTC_ClearIntFlag(RTC_INT_FLAG_TAMP1);

        /* Clear EINT RTC flag */
        EINT_ClearIntFlag(EINT_LINE19);
    }

    if (RTC_ReadIntFlag(RTC_INT_FLAG_TAMP2) == SET)
    {
        printf("RTC_INT_TAMP2\r\n");

        APM_MINI_LEDOn(LED3);

        /* Clear RTC TAMP2 flag */
        RTC_ClearIntFlag(RTC_INT_FLAG_TAMP2);

        /* Clear EINT RTC flag */
        EINT_ClearIntFlag(EINT_LINE19);
    }

    if (RTC_ReadIntFlag(RTC_INT_FLAG_TS) == SET)
    {
        RTC_ReadTimeDate(RTC_FORMAT_BIN, &timeStruct, &dateStruct);
        printf("\r\nRTC_time_Stamp : %d/%d/%d , %d-%d-%d\r\n",
               dateStruct.year, dateStruct.month, dateStruct.date,
               timeStruct.hours, timeStruct.minutes, timeStruct.seconds);

        /* Clear RTC TS flag */
        RTC_ClearIntFlag(RTC_INT_FLAG_TS);
        /* Clear EINT RTC flag */
        EINT_ClearIntFlag(EINT_LINE19);
    }
}

/*!
 * @brief       RTC_Eint Line configuration
 *
 * @param       None
 *
 * @retval      None
 *
 * @note
 */
void RTC_EintConfig(void)
{
    EINT_Config_T   eintConfig;
    GPIO_Config_T   gpioConfig;

    RCM_EnableAHBPeriphClock(RCM_AHB_PERIPH_GPIOA);
    RCM_EnableAHBPeriphClock(RCM_AHB_PERIPH_GPIOC);

    gpioConfig.pin  = GPIO_PIN_0;
    gpioConfig.mode = GPIO_MODE_IN;
    gpioConfig.pupd = GPIO_PUPD_PU;
    GPIO_Config(GPIOA, &gpioConfig);

    gpioConfig.pin  = GPIO_PIN_13;
    gpioConfig.mode = GPIO_MODE_IN;
    gpioConfig.pupd = GPIO_PUPD_PU;
    GPIO_Config(GPIOC, &gpioConfig);

    /* Config EINT Line as EINT_LINE17 and EINT_LINE19 */
    eintConfig.line = EINT_LINE19 | EINT_LINE17;
    eintConfig.mode = EINT_MODE_INTERRUPT;
    eintConfig.trigger = EINT_TRIGGER_RISING;
    eintConfig.lineCmd = ENABLE;
    EINT_Config(&eintConfig);

    NVIC_EnableIRQRequest(RTC_IRQn, 0);
}

/*!
 * @brief       RTC Configuration
 *
 * @param       None
 *
 * @retval      None
 *
 * @note
 */
void RTC_Init(void)
{
    RTC_Config_T Struct;

    /* Enable PMU periph clock */
    RCM_EnableAPB1PeriphClock(RCM_APB1_PERIPH_PMU);

    /* Enable Access Backup */
    PMU_EnableBackupAccess();

    /* Enable LSI */
    RCM_EnableLSI();

    /* wait until LSI is ready */
    while (RCM_ReadStatusFlag(RCM_FLAG_LSIRDY) == RESET);

    /* Select LSI as RTC clock */
    RCM_ConfigRTCCLK(RCM_RTCCLK_LSI);

    /* Enable RTC clock */
    RCM_EnableRTCCLK();

    RTC_DisableWriteProtection();

    RTC_WaitForSynchro();

    /* LSI */
    RTC_ConfigStructInit(&Struct);
    Struct.AsynchPrediv = 0X63;
    Struct.SynchPrediv  = 0X18F;

    RTC_Config(&Struct);
}

/*!
 * @brief       Delay
 *
 * @param       None
 *
 * @retval      None
 *
 * @note
 */
void Delay(void)
{
    volatile uint32_t delay = 0x1Fff5;

    while (delay--);
}

#if defined (__CC_ARM) || defined (__ICCARM__) || (defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050))

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       ch:  The characters that need to be send.
*
* @param       *f:  pointer to a FILE that can recording all information
*              needed to control a stream
*
* @retval      The characters that need to be send.
*
* @note
*/
int fputc(int ch, FILE* f)
{
    /* send a byte of data to the serial port */
    USART_TxData(DEBUG_USART, (uint8_t)ch);

    /* wait for the data to be send  */
    while (USART_ReadStatusFlag(DEBUG_USART, USART_FLAG_TXBE) == RESET);

    return (ch);
}

#elif defined (__GNUC__)

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       ch:  The characters that need to be send.
*
* @retval      The characters that need to be send.
*
* @note
*/
int __io_putchar(int ch)
{
    /* send a byte of data to the serial port */
    USART_TxData(DEBUG_USART, ch);

    /* wait for the data to be send  */
    while (USART_ReadStatusFlag(DEBUG_USART, USART_FLAG_TXBE) == RESET);

    return ch;
}

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       file:  Meaningless in this function.
*
* @param       *ptr:  Buffer pointer for data to be sent.
*
* @param       len:  Length of data to be sent.
*
* @retval      The characters that need to be send.
*
* @note
*/
int _write(int file, char* ptr, int len)
{
    int i;
    for (i = 0; i < len; i++)
    {
        __io_putchar(*ptr++);
    }

    return len;
}

#else
#warning Not supported compiler type
#endif

/**@} end of group RTC_Stamp_Functions */
/**@} end of group RTC_Stamp */
/**@} end of group Examples */
