/**
 * @file        apm32f4xx_adc_cfg.c
 *
 * @brief       This file provides configuration support for ADC
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_adc_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
ADC_HandleTypeDef hadc1;
DMA_HandleTypeDef hdma_adc1;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the ADC MSP
 *
 * @param  hadc pointer to a ADC_HandleTypeDef structure that contains
 *         the configuration information for the specified ADC. 
 *
 * @retval None
 */
void DAL_ADC_MspInit(ADC_HandleTypeDef* hadc)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    if (hadc->Instance == ADC1)
    {
        __DAL_RCM_ADC1_CLK_ENABLE();

        __DAL_RCM_GPIOA_CLK_ENABLE();

        /* ADC1 pin configuration */
        GPIO_InitStruct.Pin     = GPIO_PIN_0 | GPIO_PIN_1 | GPIO_PIN_2;
        GPIO_InitStruct.Mode    = GPIO_MODE_ANALOG;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* ADC1 DMA Init */
        hdma_adc1.Instance                  = DMA1_Channel1;
        hdma_adc1.Init.Direction            = DMA_PERIPH_TO_MEMORY;
        hdma_adc1.Init.PeriphInc            = DMA_PINC_DISABLE;
        hdma_adc1.Init.MemInc               = DMA_MINC_ENABLE;
        hdma_adc1.Init.PeriphDataAlignment  = DMA_PDATAALIGN_HALFWORD;
        hdma_adc1.Init.MemDataAlignment     = DMA_MDATAALIGN_HALFWORD;
        hdma_adc1.Init.Mode                 = DMA_CIRCULAR;
        hdma_adc1.Init.Priority             = DMA_PRIORITY_HIGH;
        if (DAL_DMA_Init(&hdma_adc1) != DAL_OK)
        {
            Error_Handler();
        }

        __DAL_LINKDMA(hadc,DMA_Handle,hdma_adc1);
    }
}

/**
 * @brief  DeInitializes the ADC MSP
 *
 * @param  hadc pointer to a ADC_HandleTypeDef structure that contains
 *         the configuration information for the specified ADC
 *
 * @retval None
 */
void DAL_ADC_MspDeInit(ADC_HandleTypeDef* hadc)
{
    if (hadc->Instance == ADC1)
    {
        /* Reset peripheral clock */
        __DAL_RCM_ADC1_FORCE_RESET();
        __DAL_RCM_ADC1_RELEASE_RESET();

        /* ADC GPIO configuration */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_0 | GPIO_PIN_1 | GPIO_PIN_2);

        /* Disable peripheral DMA */
        DAL_DMA_DeInit(hadc->DMA_Handle);

        /* Disable peripheral interrupt */
        DAL_NVIC_DisableIRQ(DMA1_Channel1_IRQn);
    }
}

/**
 * @brief   ADC1 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_ADC1_Config(void)
{
    ADC_ChannelConfTypeDef Channel_ConfigStruct = {0};

    /* Configure the ADC1 */
    hadc1.Instance                      = ADC1;
    hadc1.Init.ScanConvMode             = ENABLE;
    hadc1.Init.ContinuousConvMode       = ENABLE;
    hadc1.Init.DiscontinuousConvMode    = DISABLE;
    hadc1.Init.ExternalTrigConv         = ADC_SOFTWARE_START;
    hadc1.Init.DataAlign                = ADC_DATAALIGN_RIGHT;
    hadc1.Init.NbrOfConversion          = 3U;
    if (DAL_ADC_Init(&hadc1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure for the selected ADC regular channel */
    Channel_ConfigStruct.Channel        = ADC_CHANNEL_0;
    Channel_ConfigStruct.Rank           = ADC_REGULAR_RANK_1;
    Channel_ConfigStruct.SamplingTime   = ADC_SAMPLETIME_239CYCLES_5;
    if (DAL_ADC_ConfigChannel(&hadc1, &Channel_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
    
    Channel_ConfigStruct.Channel        = ADC_CHANNEL_1;
    Channel_ConfigStruct.Rank           = ADC_REGULAR_RANK_2;
    Channel_ConfigStruct.SamplingTime   = ADC_SAMPLETIME_239CYCLES_5;
    if (DAL_ADC_ConfigChannel(&hadc1, &Channel_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
    
    Channel_ConfigStruct.Channel        = ADC_CHANNEL_2;
    Channel_ConfigStruct.Rank           = ADC_REGULAR_RANK_3;
    Channel_ConfigStruct.SamplingTime   = ADC_SAMPLETIME_239CYCLES_5;
    if (DAL_ADC_ConfigChannel(&hadc1, &Channel_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}
