/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr2;
TMR_HandleTypeDef htmr3;
TMR_HandleTypeDef htmr4;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR PWM MSP
 *
 * @param  htmr TMR PWM handle
 *
 * @retval None
 */
void DAL_TMR_PWM_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR2)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR2_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;

        /* TMR Channel 1 */
        GPIO_InitStruct.Pin         = GPIO_PIN_0;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }

    if (htmr->Instance == TMR3)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR3_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;

        /* TMR Channel 1 */
        GPIO_InitStruct.Pin         = GPIO_PIN_6;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }

    if (htmr->Instance == TMR4)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR4_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOB_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;

        /* TMR channel 1 */
        GPIO_InitStruct.Pin         = GPIO_PIN_6;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
    }
}

/**
 * @brief  DeInitializes TMR PWM MSP
 *
 * @param  htmr TMR PWM handle
 *
 * @retval None
 */
void DAL_TMR_PWM_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR2)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR2_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_0);
    }

    if (htmr->Instance == TMR3)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR3_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_6);
    }

    if (htmr->Instance == TMR4)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR4_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_6);
    }
}

/**
 * @brief   TMR2 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR2_Config(void)
{
    TMR_OC_InitTypeDef OC_ConfigStruct = {0};
    TMR_MasterConfigTypeDef Master_ConfigStruct = {0};

    htmr2.Instance                  = TMR2;
    htmr2.Init.Prescaler            = 0U;
    htmr2.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr2.Init.Period               = 119U;
    htmr2.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr2.Init.RepetitionCounter    = 0U;
    htmr2.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_PWM_Init(&htmr2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure output channel */
    OC_ConfigStruct.OCMode      = TMR_OCMODE_PWM1;
    OC_ConfigStruct.OCPolarity  = TMR_OCPOLARITY_HIGH;
    OC_ConfigStruct.Pulse       = 60U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr2, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure master TMR */
    Master_ConfigStruct.MasterOutputTrigger = TMR_TRGO_UPDATE;
    Master_ConfigStruct.MasterSlaveMode     = TMR_MASTERSLAVEMODE_ENABLE;
    if (DAL_TMREx_MasterConfigSynchronization(&htmr2, &Master_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
 * @brief   TMR3 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR3_Config(void)
{
    TMR_OC_InitTypeDef OC_ConfigStruct = {0};
    TMR_SlaveConfigTypeDef Slave_ConfigStruct = {0};

    htmr3.Instance                  = TMR3;
    htmr3.Init.Prescaler            = 0U;
    htmr3.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr3.Init.Period               = 3U;
    htmr3.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr3.Init.RepetitionCounter    = 0U;
    htmr3.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_PWM_Init(&htmr3) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure output channel */
    OC_ConfigStruct.OCMode      = TMR_OCMODE_PWM1;
    OC_ConfigStruct.OCPolarity  = TMR_OCPOLARITY_HIGH;
    OC_ConfigStruct.Pulse       = 2U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr3, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure slave TMR */
    Slave_ConfigStruct.SlaveMode     = TMR_SLAVEMODE_GATED;
    Slave_ConfigStruct.InputTrigger  = TMR_TS_ITR1;
    if (DAL_TMR_SlaveConfigSynchro(&htmr3, &Slave_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
 * @brief   TMR4 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR4_Config(void)
{
    TMR_OC_InitTypeDef OC_ConfigStruct = {0};
    TMR_SlaveConfigTypeDef Slave_ConfigStruct = {0};

    htmr4.Instance                  = TMR4;
    htmr4.Init.Prescaler            = 0U;
    htmr4.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr4.Init.Period               = 1U;
    htmr4.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr4.Init.RepetitionCounter    = 0U;
    htmr4.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_PWM_Init(&htmr4) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure output channel */
    OC_ConfigStruct.OCMode      = TMR_OCMODE_PWM1;
    OC_ConfigStruct.OCPolarity  = TMR_OCPOLARITY_HIGH;
    OC_ConfigStruct.Pulse       = 1U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure slave TMR */
    Slave_ConfigStruct.SlaveMode     = TMR_SLAVEMODE_GATED;
    Slave_ConfigStruct.InputTrigger  = TMR_TS_ITR1;
    if (DAL_TMR_SlaveConfigSynchro(&htmr4, &Slave_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}
