/**
 * @file        apm32f4xx_usart_cfg.c
 *
 * @brief       This file provides configuration support for USART
 *
 * @version     V1.0.0
 *
 * @date        2024-12-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_usart_cfg.h"

/* Private includes *******************************************************/
#include <stdio.h>
#include "apm32f4xx_dal_smartcard.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
UART_HandleTypeDef huart1;
SMARTCARD_HandleTypeDef hsc3;
/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief   USART1 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_USART1_Config(void)
{
    huart1.Instance             = USART1;
    huart1.Init.BaudRate        = 115200U;
    huart1.Init.WordLength      = UART_WORDLENGTH_8B;
    huart1.Init.StopBits        = UART_STOPBITS_1;
    huart1.Init.Parity          = UART_PARITY_NONE;
    huart1.Init.Mode            = UART_MODE_TX_RX;
    huart1.Init.HwFlowCtl       = UART_HWCONTROL_NONE;
    huart1.Init.OverSampling    = UART_OVERSAMPLING_16;
    
    if (DAL_UART_Init(&huart1) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
 * @brief   USART1 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_USART3_SMARTCARD_Config(void)
{
    hsc3.Instance             = USART3;
    hsc3.Init.BaudRate        = 9677;
    hsc3.Init.WordLength      = SMARTCARD_WORDLENGTH_9B;
    hsc3.Init.StopBits        = SMARTCARD_STOPBITS_1_5;
    hsc3.Init.Parity          = SMARTCARD_PARITY_EVEN;
    hsc3.Init.Mode            = SMARTCARD_MODE_TX_RX;
    hsc3.Init.CLKLastBit      = SMARTCARD_LASTBIT_ENABLE;
    hsc3.Init.CLKPhase        = SMARTCARD_PHASE_1EDGE;
    hsc3.Init.CLKPolarity     = SMARTCARD_POLARITY_LOW;
    hsc3.Init.Prescaler       = SMARTCARD_PRESCALER_SYSCLK_DIV10;
    hsc3.Init.NACKState       = SMARTCARD_NACK_ENABLE;
    hsc3.Init.GuardTime       = 16;
    
    if (DAL_SMARTCARD_Init(&hsc3) != DAL_OK)
    {
        Error_Handler();
    }
    
    __DAL_SMARTCARD_ENABLE_IT(&hsc3, SMARTCARD_IT_PE);
}

/**
 * @brief   UART MSP Init
 *
 * @param   huart  Pointer to a UART_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval  None
 */
void DAL_UART_MspInit(UART_HandleTypeDef *huart)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    if (huart->Instance == USART1)
    {
        /* Enable USART1 GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();
        
        /* Enable USART1 clock */
        __DAL_RCM_USART1_CLK_ENABLE();

        /* Configure the UART TX pin */
        GPIO_InitStruct.Pin         = GPIO_PIN_9;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;

        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* Configure the UART RX pin */
        GPIO_InitStruct.Pin         = GPIO_PIN_10;
        GPIO_InitStruct.Mode        = GPIO_MODE_INPUT;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;

        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }
}

/**
 * @brief   SMARTCARD MSP Init
 *
 * @param   huart  Pointer to a SMARTCARD_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval  None
 */
void DAL_SMARTCARD_MspInit(SMARTCARD_HandleTypeDef* hsc)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    if (hsc->Instance == USART3)
    {
        /* Enable USART3 GPIO clock */
        __DAL_RCM_GPIOB_CLK_ENABLE();
        
        /* Enable USART3 clock */
        __DAL_RCM_USART3_CLK_ENABLE();

        /* Configure the GPIO ports */
        /* USART3 GPIO Configuration
           PB10 (USART3_TX) ---> Smartcard Tx Pin
           PB11 (Pin)       ---> Smartcard Reset Pin
           PB12 (USART3_CK) ---> Smartcard CK Pin
        */

        /* Configure Smartcard USART (USART3) CK as alternate function push-pull */
        GPIO_InitStruct.Pin         = GPIO_PIN_12;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        /* Configure Smartcard USART (USART3) Tx as alternate function open-drain */
        GPIO_InitStruct.Pin         = GPIO_PIN_10;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_OD;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        /* Configure Smartcard Reset Pin */
        GPIO_InitStruct.Pin         = GPIO_PIN_11;
        GPIO_InitStruct.Mode        = GPIO_MODE_OUTPUT_OD;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        /* USART3 interrupt Init */
        DAL_NVIC_SetPriority(USART3_IRQn, 1, 0);
        DAL_NVIC_EnableIRQ(USART3_IRQn);

        DAL_GPIO_WritePin(GPIOB, GPIO_PIN_11, GPIO_PIN_SET);
    }
}

/**
 * @brief   SMARTCARD MSP DeInit
 *
 * @param   huart  Pointer to a SMARTCARD_HandleTypeDef structure that contains
 *                the configuration information for the specified UART module
 *
 * @retval  None
 */
void DAL_SMARTCARD_MspDeInit(SMARTCARD_HandleTypeDef* hsc)
{
    if (hsc->Instance == USART3)
    {
        /* Reset USART */
        __DAL_RCM_USART3_FORCE_RESET();
        __DAL_RCM_USART3_RELEASE_RESET();
        
        /* Disable USART and GPIO clocks */
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_10);
        
        /* USART3 interrupt Deinit */
        DAL_NVIC_DisableIRQ(USART3_IRQn);
    }
}

/**
  * @brief  UART MSP DeInit
  *
  * @param  huart  Pointer to a UART_HandleTypeDef structure that contains
  *                the configuration information for the specified UART module
  *
  * @retval None
  */
void DAL_UART_MspDeInit(UART_HandleTypeDef *huart)
{
    if (huart->Instance == USART1)
    {
        /* Reset USART */
        __DAL_RCM_USART1_FORCE_RESET();
        __DAL_RCM_USART1_RELEASE_RESET();
        
        /* Disable USART and GPIO clocks */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_9 | GPIO_PIN_10);
    }
}

#if defined (__CC_ARM) || defined (__ICCARM__) || (defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050))

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       ch:  The characters that need to be send.
*
* @param       *f:  pointer to a FILE that can recording all information
*              needed to control a stream
*
* @retval      The characters that need to be send.
*
* @note
*/
int fputc(int ch, FILE* f)
{
    /* send a byte of data to the serial port */
    DAL_UART_Transmit(&huart1, (uint8_t *)&ch, 1U, 1000U);

    return (ch);
}

#elif defined (__GNUC__)

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       ch:  The characters that need to be send.
*
* @retval      The characters that need to be send.
*
* @note
*/
int __io_putchar(int ch)
{
    /* send a byte of data to the serial port */
    DAL_UART_Transmit(&huart1, (uint8_t *)&ch, 1U, 1000U);

    return ch;
}

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       file:  Meaningless in this function.
*
* @param       *ptr:  Buffer pointer for data to be sent.
*
* @param       len:  Length of data to be sent.
*
* @retval      The characters that need to be send.
*
* @note
*/
int _write(int file, char* ptr, int len)
{
    int i;
    for (i = 0; i < len; i++)
    {
        __io_putchar(*ptr++);
    }

    return len;
}

#else
#warning Not supported compiler type
#endif
