/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2024-08-01
 *
 * @attention
 *
 *  Copyright (C) 2024-2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"

/* Private macro **********************************************************/

/* Define firmware to switch between Master board or Slave board */
#define IS_MASTER_BOARD

#define BUFFER_SIZE    32

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";

uint8_t Write_Buffer[BUFFER_SIZE] = {0};
uint8_t Read_Buffer[BUFFER_SIZE] = {0};

uint32_t size = sizeof(uint8_t);

/* Private function prototypes ********************************************/
static void InitBuffer(void);
static uint8_t CompareBuffer(uint8_t* buf1, uint8_t* buf2, uint32_t size);

/* External variables *****************************************************/
extern SPI_HandleTypeDef hspi1;
extern SPI_HandleTypeDef hspi2;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    SPI_HandleTypeDef *spi = &hspi1;
    spi->pTxBuffPtr = (uint8_t *)Write_Buffer;
    spi->pRxBuffPtr += 1;
    
    /* Initailize write buffer and read buffer */
    InitBuffer();

#if defined(IS_MASTER_BOARD)
    DAL_SPI_TransmitReceive(&hspi1, (uint8_t*)Write_Buffer, (uint8_t*)Read_Buffer, BUFFER_SIZE, 1000U);
#else
    DAL_SPI_TransmitReceive(&hspi2, (uint8_t*)Write_Buffer, (uint8_t*)Read_Buffer, BUFFER_SIZE, 1000U);
#endif

    if (CompareBuffer((uint8_t*)Write_Buffer, (uint8_t*)Read_Buffer, BUFFER_SIZE))
    {
        BOARD_LED_On(LED2);
        LOGI(tag, "SPI transmit success!\r\n");
    }
    else
    {
        BOARD_LED_On(LED3);
        LOGI(tag, "SPI transmit fail!\r\n");
    }

    /* Infinite loop */
    while (1)
    {
    }
}

/*!
 * @brief   Initialize SPI write and read buffers
 *
 * @param   None
 *
 * @retval  None
 */
static void InitBuffer(void)
{
    for(uint32_t idx = 0U; idx < BUFFER_SIZE; idx++)
    {
        Write_Buffer[idx] = (uint8_t)idx;
        Read_Buffer[idx] = 0U;
    }
}

/*!
 * @brief   Compares two buffers
 *
 * @param   buf1:    First buffer to be compared
 *
 * @param   buf1:    Second buffer to be compared
 *
 * @param   size:    Buffer size
 *
 * @retval  Return 1 if buf1 = buf2. If not then return 0
 */
static uint8_t CompareBuffer(uint8_t* buf1, uint8_t* buf2, uint32_t size)
{
    uint32_t i = 0;

    for (i = 0; i < size; i++)
    {
        if (buf1[i] != buf2[i])
        {
            return 0;
        }
    }

    return 1;
}
