/**
 * @file        lcd.h
 *
 * @brief       This file provides application support for LCD
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Define to prevent recursive inclusion */
#ifndef LCD_H
#define LCD_H

#ifdef __cplusplus
  extern "C" {
#endif

/* Includes ***************************************************************/
#include "apm32f4xx_dal.h"
#include "lcd_font.h"

/* Exported macro *********************************************************/

/**
 * @brief LCD_Status LCD status
 */
#define LCD_STATUS_OK                           0x00U
#define LCD_STATUS_ERROR                        0x01U
#define LCD_STATUS_TIMEOUT                      0x02U

/**
 * @brief LCD_ScanMode LCD scan mode
 */
#define LCD_SCAN_MODE_0                         0x00U
#define LCD_SCAN_MODE_1                         0x01U
#define LCD_SCAN_MODE_2                         0x02U
#define LCD_SCAN_MODE_3                         0x03U
#define LCD_SCAN_MODE_4                         0x04U
#define LCD_SCAN_MODE_5                         0x05U
#define LCD_SCAN_MODE_6                         0x06U
#define LCD_SCAN_MODE_7                         0x07U


/**
 * @brief LCD Long and short side
 */
#define LCD_SIDE_SHORT                          (480U)
#define LCD_SIDE_LONG                           (800U)

/* SMC Address 
 * 0x60000000 - 0x63FFFFFF: SMC_Bank1_NOR/SRAM1
 * 0x64000000 - 0x67FFFFFF: SMC_Bank1_NOR/SRAM2
 * 0x68000000 - 0x6BFFFFFF: SMC_Bank1_NOR/SRAM3
 * 0x6C000000 - 0x6FFFFFFF: SMC_Bank1_NOR/SRAM4
 * 
 * LCD CS -> SMC_NE3 -> SMC_Bank1_NOR/SRAM3
 * LCD RS -> SMC_A10
 */
#define LCD_ADDR_SMC_PIN                        (10)
#define LCD_CMD_SMC_ADDR                        (0x68000000)
#define LCD_DATA_SMC_ADDR                       (0x68000000 | (1 << (LCD_ADDR_SMC_PIN + 1)))

/**
 * @brief LCD Color
 */
#define LCD_COLOR_WHITE                         ((uint16_t)0xFFFF)
#define LCD_COLOR_BLACK                         ((uint16_t)0x0000)
#define LCD_COLOR_RED                           ((uint16_t)0xF800)
#define LCD_COLOR_GRED                          ((uint16_t)0XFFE0)
#define LCD_COLOR_BRED                          ((uint16_t)0XF81F)
#define LCD_COLOR_BRRED                         ((uint16_t)0XFC07)
#define LCD_COLOR_GREEN                         ((uint16_t)0x07E0)
#define LCD_COLOR_BLUE                          ((uint16_t)0x001F)
#define LCD_COLOR_GBLUE                         ((uint16_t)0X07FF)
#define LCD_COLOR_YELLOW                        ((uint16_t)0xFFE0)
#define LCD_COLOR_MAGENTA                       ((uint16_t)0xF81F)
#define LCD_COLOR_CYAN                          ((uint16_t)0x7FFF)
#define LCD_COLOR_BROWN                         ((uint16_t)0XBC40)
#define LCD_COLOR_GRAY                          ((uint16_t)0X8430)

#define LCD_R                                   1    //!< User define, must be less than 31
#define LCD_G                                   60   //!< User define, must be less than 63
#define LCD_B                                   23   //!< User define, must be less than 31
#define LCD_COLOR_USER                          ((uint16_t)((LCD_R << 11 & 0xF800) | (LCD_G << 5 & 0x07E0) | (LCD_B & 0x001F)))

#define LCD_COLOR_BACKGROUND                    LCD_COLOR_WHITE   //!< Default background color
#define LCD_COLOR_FOREGROUND                    LCD_COLOR_USER    //!< Default foreground color                       

/* Exported typedef *******************************************************/

/**
 * @brief LCD_Informations LCD informations
 */
typedef struct
{
    uint16_t width;             /*!< Width */
    uint16_t height;            /*!< Height */
    uint16_t scanMode;          /*!< Scan mode */
    uint16_t backColor;         /*!< Background color */
    uint32_t lastWindowArea;    /*!< Last window area */
    LCD_FONT_T *font;           /*!< Font information */
} LCD_INFO_T;

/* Exported function prototypes *******************************************/
uint32_t LCD_Init(void);
uint32_t LCD_DeInit(void);
void LCD_Reset(void);
void LCD_SetBacklight(FunctionalState state);
void LCD_SetScanMode(uint16_t scanMode);
void LCD_SetBackColor(uint16_t color);
uint16_t LCD_GetBackColor(void);
uint32_t LCD_GetMaxX(void);
uint32_t LCD_GetMaxY(void);
void LCD_OpenWindow(uint16_t x, uint16_t y, uint16_t xSize, uint16_t ySize);
void LCD_FillWindow(uint16_t color);
void LCD_ClearWindow(void);
void LCD_PutString(uint16_t x, uint16_t y, char* str, uint16_t color);
void LCD_PutChar(uint16_t x, uint16_t y, uint8_t charCode, uint16_t color);
void LCD_DrawPoint(uint16_t x, uint16_t y, uint16_t color);
void LCD_DrawLine(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1, uint16_t color);
void LCD_DrawRectangle(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1, uint16_t color);
void LCD_DrawCircle(uint16_t x0, uint16_t y0, uint16_t r, uint16_t color, FunctionalState fill);

#ifdef __cplusplus
}
#endif

#endif /* LCD_H */
