/**
 * @file        lcd.c
 *
 * @brief       This file provides application support for LCD
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "lcd.h"

/* Private includes *******************************************************/
#include "board_nt35510.h"
#include "apm32f4xx_device_cfg.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* External variables *****************************************************/
extern LCD_FONT_T fontInfo;

/* Private variables ******************************************************/
LCD_INFO_T lcdInfo = {
    .width = LCD_SIDE_SHORT,
    .height = LCD_SIDE_LONG,
    .scanMode = LCD_SCAN_MODE_6,
    .backColor = LCD_COLOR_BACKGROUND,
    .lastWindowArea = LCD_SIDE_LONG * LCD_SIDE_SHORT,
    .font = &fontInfo,
};

/* Private function prototypes ********************************************/

/* External functions *****************************************************/

/**
 * @brief  Initialize LCD.
 * @param  None
 * @retval LCD status
 */
uint32_t LCD_Init(void)
{
    uint32_t status = LCD_STATUS_OK;

    /* Initialize LCD */
    status = NT35510_Init();

    if (status == NT35510_STATUS_OK)
    {
        /* Reset LCD */
        LCD_Reset();

        status = NT35510_RegConfig();
        
        if (status == NT35510_STATUS_OK)
        {
            /* Set LCD backlight state */
            LCD_SetBacklight(ENABLE);

            /* Set LCD scan mode */
            LCD_SetScanMode(lcdInfo.scanMode);

            /* Set LCD background color */
            LCD_SetBackColor(lcdInfo.backColor);

            /* Clear LCD window */
            LCD_ClearWindow();
        }
    }

    return status;
}

/**
 * @brief  Deinitialize LCD.
 * @param  None
 * @retval LCD status
 */
uint32_t LCD_DeInit(void)
{
    uint32_t status = LCD_STATUS_OK;

    /* Deinitialize LCD */
    status = NT35510_DeInit();

    return status;
}

/**
 * @brief  Reset LCD.
 * 
 * @param  None
 * 
 * @retval None
 */
void LCD_Reset(void)
{
    DAL_GPIO_WritePin(LCD_RST_GPIO_PORT, LCD_RST_GPIO_PIN, GPIO_PIN_RESET);
    DAL_Delay(500);
    DAL_GPIO_WritePin(LCD_RST_GPIO_PORT, LCD_RST_GPIO_PIN, GPIO_PIN_SET);
    DAL_Delay(500);
}

/**
 * @brief  Set LCD backlight state.
 * 
 * @param  state backlight state
 * 
 * @retval None
 */
void LCD_SetBacklight(FunctionalState state)
{
    if (state == ENABLE)
    {
        DAL_GPIO_WritePin(LCD_BK_GPIO_PORT, LCD_BK_GPIO_PIN, GPIO_PIN_SET);
    }
    else
    {
        DAL_GPIO_WritePin(LCD_BK_GPIO_PORT, LCD_BK_GPIO_PIN, GPIO_PIN_RESET);
    }
}

/**
 * @brief  Set LCD scan mode.
 * 
 * @param  scanMode scan mode
 *          This parameter can be one of the following values:
 *          @arg LCD_SCAN_MODE_0 : axis-X and axis-Y on screen:
 *                                  (y)^
 *                                      |
 *                                      |
 *                              (x)<---0

 *          @arg LCD_SCAN_MODE_1 : axis-X and axis-Y on screen:
 *                                  (x)^
 *                                      |
 *                                      |
 *                              (y)<---0

 *          @arg LCD_SCAN_MODE_2 : axis-X and axis-Y on screen:
 *                              ^(y)
 *                              |
 *                              |
 *                              0--->(x)

 *          @arg LCD_SCAN_MODE_3 : axis-X and axis-Y on screen:
 *                              ^(x)
 *                              |
 *                              |
 *                              0--->(y)

 *          @arg LCD_SCAN_MODE_4 : axis-X and axis-Y on screen:
 *                              (x)<---0
 *                                      |
 *                                      |
 *                                      V(y)

 *          @arg LCD_SCAN_MODE_5 : axis-X and axis-Y on screen:
 *                              (y)<---0
 *                                      |
 *                                      |
 *                                      V(x)

 *          @arg LCD_SCAN_MODE_6 : axis-X and axis-Y on screen:
 *                              0--->(x)
 *                              |
 *                              |
 *                              V(y)

 *          @arg LCD_SCAN_MODE_7 : axis-X and axis-Y on screen:
 *                              0--->(y)
 *                              |
 *                              |
 *                              V(x)
 *
 * @retval None
 */
void LCD_SetScanMode(uint16_t scanMode)
{
    /* Memory Data Access Control */
    NT35510_WriteCmdDataCallback(0x3600, scanMode << 5);

    /* Row/Column Exchange */
    if (scanMode & 0x01)
    {
        lcdInfo.width = LCD_SIDE_LONG;
        lcdInfo.height = LCD_SIDE_SHORT;
    }
    else
    {
        lcdInfo.width = LCD_SIDE_SHORT;
        lcdInfo.height = LCD_SIDE_LONG;
    }
    lcdInfo.scanMode = scanMode;

    /* Column Address Set */
    NT35510_WriteCmdDataCallback(0x2A00, 0x00);
    NT35510_WriteCmdDataCallback(0x2A01, 0x00);
    NT35510_WriteCmdDataCallback(0x2A02, ((lcdInfo.width - 1) >> 8) & 0xFF);
    NT35510_WriteCmdDataCallback(0x2A03, (lcdInfo.width - 1) & 0xFF);

    /* Row Address Set */
    NT35510_WriteCmdDataCallback(0x2B00, 0x00);
    NT35510_WriteCmdDataCallback(0x2B01, 0x00);
    NT35510_WriteCmdDataCallback(0x2B02, ((lcdInfo.height - 1) >> 8) & 0xFF);
    NT35510_WriteCmdDataCallback(0x2B03, (lcdInfo.height - 1) & 0xFF);

    /* Write Memory */
    NT35510_WriteCmdCallback(0x2C00);
}

/**
 * @brief  Open LCD window to display.
 * 
 * @param  x: X position
 * 
 * @param  y: Y position
 * 
 * @param  xSize: X size
 * 
 * @param  ySize: Y size
 * 
 * @retval None
 */
void LCD_OpenWindow(uint16_t x, uint16_t y, uint16_t xSize, uint16_t ySize)
{
    /* Column Address Set */
    NT35510_WriteCmdDataCallback(0x2A00, x >> 8);
    NT35510_WriteCmdDataCallback(0x2A01, x & 0xFF);
    NT35510_WriteCmdDataCallback(0x2A02, (x + xSize - 1) >> 8);
    NT35510_WriteCmdDataCallback(0x2A03, (x + xSize - 1) & 0xFF);

    /* Row Address Set */
    NT35510_WriteCmdDataCallback(0x2B00, y >> 8);
    NT35510_WriteCmdDataCallback(0x2B01, y & 0xFF);
    NT35510_WriteCmdDataCallback(0x2B02, (y + ySize - 1) >> 8);
    NT35510_WriteCmdDataCallback(0x2B03, (y + ySize - 1) & 0xFF);

    lcdInfo.lastWindowArea = xSize * ySize;
}

/**
 * @brief  Fill LCD window with color.
 * 
 * @param  color: Color to fill
 * 
 * @retval None
 */
void LCD_FillWindow(uint16_t color)
{
    uint32_t i = 0;

    /* Write Memory */
    NT35510_WriteCmdCallback(0x2C00);

    for (i = 0; i < lcdInfo.lastWindowArea; i++)
    {
        NT35510_WriteDataCallback(color);
    }
}

/**
 * @brief  Clear LCD window.
 * 
 * @param  None
 * 
 * @retval None
 */
void LCD_ClearWindow(void)
{
    if (lcdInfo.scanMode & 0x01)
    {
        LCD_OpenWindow(0, 0, LCD_SIDE_LONG, LCD_SIDE_SHORT);
    }
    else
    {
        LCD_OpenWindow(0, 0, LCD_SIDE_SHORT, LCD_SIDE_LONG);
    }

    LCD_FillWindow(lcdInfo.backColor);
}

/**
 * @brief  Set LCD background color.
 * 
 * @param  color background color
 * 
 * @retval None
 */
void LCD_SetBackColor(uint16_t color)
{
    lcdInfo.backColor = color;
}

/**
 * @brief  Get LCD background color.
 * 
 * @param  None
 * 
 * @retval LCD background color
 */
uint16_t LCD_GetBackColor(void)
{
    return lcdInfo.backColor;
}

/**
 * @brief  Get max length of X axis in scan mode now.
 * 
 * @param  None
 * 
 * @retval Max length of X axis
 */
uint32_t LCD_GetMaxX(void)
{
    return (lcdInfo.scanMode & 0x01) ? LCD_SIDE_LONG : LCD_SIDE_SHORT;
}

/**
 * @brief  Get max length of Y axis in scan mode now.
 * 
 * @param  None
 * 
 * @retval Max length of Y axis
 */
uint32_t LCD_GetMaxY(void)
{
    return (lcdInfo.scanMode & 0x01) ? LCD_SIDE_SHORT : LCD_SIDE_LONG;
}

/**
 * @brief  Print a character on LCD.
 * 
 * @param  x: X position
 * 
 * @param  y: Y position
 * 
 * @param  charCode: Character ascii code
 * 
 * @param  color: Character color
 * 
 * @retval None
 */
void LCD_PutChar(uint16_t x, uint16_t y, uint8_t charCode, uint16_t color)
{
    uint8_t index, data;
    uint16_t tableOffset;
    uint8_t* pTable;
    uint32_t i;

    if ((charCode < ' ') || \
        ((x > lcdInfo.width - lcdInfo.font->width)) || \
        (y > lcdInfo.height - lcdInfo.font->height))
    {
        return;
    }
    else
    {
        tableOffset = (charCode - ' ');
        tableOffset *= ((lcdInfo.font->height * lcdInfo.font->width) >> 3);
    }

    pTable = &lcdInfo.font->pTable[tableOffset];

    LCD_OpenWindow(x, y, lcdInfo.font->width, lcdInfo.font->height);
    NT35510_WriteCmdCallback(0x2C00);

    for (i = 0; i < (lcdInfo.font->width* lcdInfo.font->height) >> 3; i++)
    {
        data = *pTable++;
        index = 8;

        while (index--)
        {
            if ((data >> index) & 0x1)
            {
                NT35510_WriteDataCallback(color);
            }
            else
            {
                NT35510_WriteDataCallback(lcdInfo.backColor);
            }
        }
    }
}

/**
 * @brief  Print a string on LCD.
 * 
 * @param  x: X position
 * 
 * @param  y: Y position
 * 
 * @param  str: String to print
 * 
 * @param  color: String color
 * 
 * @retval None
 */
void LCD_PutString(uint16_t x, uint16_t y, char* str, uint16_t color)
{
    uint16_t x0 = x;

    while (*str != '\0')
    {
        if (*str == '\n')
        {
            x = x0;
            y += lcdInfo.font->height;
        }
        else
        {
            LCD_PutChar(x, y, *str, color);
            x += lcdInfo.font->width;
        }

        str++;
    }
}

/**
 * @brief  Draw a point on LCD.
 * 
 * @param  x: X position
 * 
 * @param  y: Y position
 * 
 * @param  color: Point color
 * 
 * @retval None
 */
void LCD_DrawPoint(uint16_t x, uint16_t y, uint16_t color)
{
    if ((x > lcdInfo.width) || (y > lcdInfo.height))
    {
        return;
    }

    LCD_OpenWindow(x, y, 1, 1);
    LCD_FillWindow(color);
}

/**
 * @brief  Draw a line on LCD.
 * 
 * @param  x0: X start position
 * 
 * @param  y0: Y start position
 * 
 * @param  x1: X end position
 * 
 * @param  y1: Y end position
 * 
 * @param  color: Line color
 * 
 * @retval None
 */
void LCD_DrawLine(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1, uint16_t color)
{
    uint16_t xLength = (x0 > x1) ? (x0 - x1) : (x1 - x0);
    uint16_t yLength = (y0 > y1) ? (y0 - y1) : (y1 - y0);
    int32_t xError = 0, yError = 0, xAdd = 0, yAdd = 0, length; 
    uint32_t i;

    /* Increase or Decrease */
    if (x0 != x1)
    {
        xAdd = (x0 > x1) ? -1 : 1;
    }

    /* Increase or Decrease */
    if (y0 != y1)
    {
        yAdd = (y0 > y1) ? -1 : 1;
    }

    if (xLength > yLength)
    {
        length = xLength;
    }
    else
    {
        length = yLength;
    }

    /* Draw Line */
    for (i = 0; i <= length + 1; i ++ )
    {  
        LCD_DrawPoint( x0, y0, color);

        xError += xLength;
        yError += yLength;

        if (xError > length)
        { 
            xError -= length;
            x0 += xAdd;
        }

        if (yError > length) 
        { 
            yError -= length;
            y0 += yAdd;
        }
    }
}

/**
 * @brief  Draw a rectangle on LCD.
 * 
 * @param  x0: X start position
 * 
 * @param  y0: Y start position
 * 
 * @param  x1: X end position
 * 
 * @param  y1: Y end position
 * 
 * @param  color: Rectangle color
 * 
 * @retval None
 */
void LCD_DrawRectangle(uint16_t x0, uint16_t y0, uint16_t x1, uint16_t y1, uint16_t color)
{
    LCD_DrawLine(x0, y0, x1, y0, color);
    LCD_DrawLine(x0, y0, x0, y1, color);
    LCD_DrawLine(x1, y1, x1, y0, color);
    LCD_DrawLine(x1, y1, x0, y1, color);
}

/**
 * @brief  Draw a circle on LCD.
 * 
 * @param  x0: X center position
 * 
 * @param  y0: Y center position
 * 
 * @param  r: Radius
 * 
 * @param  color: Circle color
 * 
 * @param  fill: Fill or not
 * 
 * @retval None
 */
void LCD_DrawCircle(uint16_t x0, uint16_t y0, uint16_t r, uint16_t color, FunctionalState fill)
{
    int32_t x = -r, y = 0, err = 2 - 2 * r, e2;

    do
    {
        if (fill == ENABLE)
        {
            LCD_DrawLine(x0 - x, y0 + y, x0 + x, y0 + y, color);
            LCD_DrawLine(x0 - x, y0 - y, x0 + x, y0 - y, color);
        }
        else
        {
            LCD_DrawPoint(x0 - x, y0 + y, color);
            LCD_DrawPoint(x0 + x, y0 + y, color);
            LCD_DrawPoint(x0 - x, y0 - y, color);
            LCD_DrawPoint(x0 + x, y0 - y, color);
        }

        e2 = err;
        if (e2 <= y)
        {
            err += ++y * 2 + 1;
        }

        if (e2 > x || err > y)
        {
            err += ++x * 2 + 1;
        }
    }
    while (x <= 0);
}

/* Callback functions ******************************************************/

/**
 * @brief  NT35510 delay
 * 
 * @param  delay: Delay in ms
 * 
 * @retval None
 */
void NT35510_DelayCallback(uint32_t delay)
{
    DAL_Delay(delay);
}

/**
 * @brief  Initialize the NT35510 interface
 *
 * @param  None
 *
 * @retval 0 if OK, -1 if ERROR
 */
int32_t NT35510_IO_InitCallback(void)
{
    DAL_SMC_Config();

    return 0;
}

/**
 * @brief  Write command to the NT35510
 * 
 * @param  cmd: Command to write
 * 
 * @retval None
 */
void NT35510_WriteCmdCallback(uint16_t cmd)
{
    *(__IO uint16_t*)(LCD_CMD_SMC_ADDR) = cmd;
}

/**
 * @brief  Write data to the NT35510
 * 
 * @param  data: Data to write
 * 
 * @retval None
 */
void NT35510_WriteDataCallback(uint16_t data)
{
    *(__IO uint16_t*)(LCD_DATA_SMC_ADDR) = data;
}

/**
 * @brief  Write command and data to the NT35510
 * 
 * @param  cmd: Command to write
 * 
 * @param  data: Data to write
 * 
 * @retval None
 */
void NT35510_WriteCmdDataCallback(uint16_t cmd, uint16_t data)
{
    *(__IO uint16_t*)(LCD_CMD_SMC_ADDR) = cmd;
    *(__IO uint16_t*)(LCD_DATA_SMC_ADDR) = data;
}

/**
 * @brief  Read data from the NT35510
 * 
 * @param  None
 * 
 * @retval Read data
 */
uint16_t NT35510_ReadDataCallback(void)
{
    return (*(__IO uint16_t*)(LCD_DATA_SMC_ADDR));
}
