/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"
#include <string.h>

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";

/* Plaintext */
uint32_t plaintext[16] = 
{
    0x6BC1BEE2U, 0x2E409F96U, 0xE93D7E11U, 0x7393172AU,
    0xAE2D8A57U, 0x1E03AC9CU, 0x9EB76FACU, 0x45AF8E51U,
    0x30C81C46U, 0xA35CE411U, 0xE5FBC119U, 0x1A0A52EFU,
    0xF69F2445U, 0xDF4F9B17U, 0xAD2B417BU, 0xE66C3710U
};

/* Expected ECB ciphertext with AES128Key */
uint32_t ciphertextAES128ECB[16] = 
{
    0x3AD77BB4U, 0x0D7A3660U, 0xA89ECAF3U, 0x2466EF97U,
    0xF5D3D585U, 0x03B9699DU, 0xE785895AU, 0x96FDBAAFU,
    0x43B1CD7FU, 0x598ECE23U, 0x881B00E3U, 0xED030688U,
    0x7B0C785EU, 0x27E8AD3FU, 0x82232071U, 0x04725DD4U
};

/* Expected CBC ciphertext with AES192Key*/
uint32_t ciphertextAES192CBC[16] = 
{
    0x4F021DB2U, 0x43BC633DU, 0x7178183AU, 0x9FA071E8U,
    0xB4D9ADA9U, 0xAD7DEDF4U, 0xE5E73876U, 0x3F69145AU,
    0x571B2420U, 0x12FB7AE0U, 0x7FA9BAACU, 0x3DF102E0U,
    0x08B0E279U, 0x88598881U, 0xD920A9E6U, 0x4F5615CDU
};

/* Expected CTR ciphertext with AES256Key */
uint32_t ciphertextAES256CTR[16] = 
{
    0x601EC313U, 0x775789A5U, 0xB7A7F504U, 0xBBF3D228U,
    0xF443E3CAU, 0x4D62B59AU, 0xCA84E990U, 0xCACAF5C5U,
    0x2B0930DAU, 0xA23DE94CU, 0xE87017BAU, 0x2D84988DU,
    0xDFC9C58DU, 0xB67AADA6U, 0x13C2DD08U, 0x457941A6U
};

/* Encrypted text buffer */
static uint32_t encryptedText[16] = {0};

/* Decrypted text buffer */
static uint32_t decryptedText[16] = {0};

/* Private function prototypes ********************************************/

/* External variables *****************************************************/
extern CRYP_HandleTypeDef hcryp;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* AES mode ECB */
    DAL_CRYP_AESECBConfig();

    /* Encrypt the plaintext message*/
    DAL_CRYP_Encrypt(&hcryp, plaintext, 16U, encryptedText, 0xFFU);

    /* Compare with expected result */
    if(memcmp(encryptedText, ciphertextAES128ECB, 64U) != 0U)
    {
        LOGI(tag, "AES 128 ECB encryption error\r\n");
    }
    else
    {
        LOGI(tag, "AES 128 ECB encryption passed\r\n");
    }

    /* Decrypt the plaintext message */
    DAL_CRYP_Decrypt(&hcryp, ciphertextAES128ECB , 16U, decryptedText, 0xFFU);

    /*compare with expected result */
    if(memcmp(decryptedText, plaintext, 64U) != 0U)
    {
        LOGI(tag, "AES 128 ECB decryption error\r\n");
    }
    else
    {
        LOGI(tag, "AES 128 ECB decryption passed\r\n");
    }

    /* AES mode CBC */
    DAL_CRYP_AESCBCConfig();
    
    /* Encrypt the plaintext message*/
    DAL_CRYP_Encrypt(&hcryp, plaintext, 16U, encryptedText, 0xFFU);

    /* Compare with expected result */
    if(memcmp(encryptedText, ciphertextAES192CBC, 64U) != 0U)
    {
        LOGI(tag, "AES 192 CBC encryption error\r\n");
    }
    else
    {
        LOGI(tag, "AES 192 CBC encryption passed\r\n");
    }
    
    /* Decrypt the plaintext message */
    DAL_CRYP_Decrypt(&hcryp, ciphertextAES192CBC , 16U, decryptedText, 0xFFU);

    /*compare with expected result */
    if(memcmp(decryptedText, plaintext, 64U) != 0U)
    {
        LOGI(tag, "AES 192 CBC decryption error\r\n");
    }
    else
    {
        LOGI(tag, "AES 192 CBC decryption passed\r\n");
    }
    
    /* AES mode CTR */
    DAL_CRYP_AESCTRConfig();
    
    /* Encrypt the plaintext message*/
    DAL_CRYP_Encrypt(&hcryp, plaintext, 16U, encryptedText, 0xFFU);

    /* Compare with expected result */
    if(memcmp(encryptedText, ciphertextAES256CTR, 64U) != 0U)
    {
        LOGI(tag, "AES 256 CTR encryption error\r\n");
    }
    else
    {
        LOGI(tag, "AES 256 CTR encryption passed\r\n");
    }
    
    /* Decrypt the plaintext message */
    DAL_CRYP_Decrypt(&hcryp, ciphertextAES256CTR , 16U, decryptedText, 0xFFU);

    /*compare with expected result */
    if(memcmp(decryptedText, plaintext, 64U) != 0U)
    {
        LOGI(tag, "AES 256 CTR decryption error\r\n");
    }
    else
    {
        LOGI(tag, "AES 256 CTR decryption passed\r\n");
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500U);
    }
}
