/**
 * @file        apm32f4xx_i2s_cfg.c
 *
 * @brief       This file provides configuration support for I2S
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_i2s_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
I2S_HandleTypeDef hi2s2;
I2S_HandleTypeDef hi2s3;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief   I2S MSP Init
 *
 * @param   hi2s pointer to a I2S_HandleTypeDef structure that contains
 *          the configuration information for I2S module
 *
 * @retval  None
 */
void DAL_I2S_MspInit(I2S_HandleTypeDef *hi2s)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    if(hi2s->Instance == SPI2)
    {
        /* Enable peripheral clock */
        __DAL_RCM_SPI2_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOB_CLK_ENABLE();

        /* Configure the SPI pin */
        GPIO_InitStruct.Pin         = GPIO_PIN_12 | GPIO_PIN_13 | GPIO_PIN_15;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF5_SPI2;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
    }

    if(hi2s->Instance == SPI3)
    {
        /* Enable peripheral clock */
        __DAL_RCM_SPI3_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();
        __DAL_RCM_GPIOB_CLK_ENABLE();

        /* Configure the SPI pin */
        GPIO_InitStruct.Pin         = GPIO_PIN_15;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF6_SPI3;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_3 | GPIO_PIN_5;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_NOPULL;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FREQ_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF6_SPI3;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
    }
}

/**
 * @brief   I2S MSP DeInit
 *
 * @param   hi2s pointer to a I2S_HandleTypeDef structure that contains
 *          the configuration information for I2S module
 *
 * @retval  None
 */
void DAL_I2S_MspDeInit(I2S_HandleTypeDef *hi2s)
{
    if(hi2s->Instance == SPI2)
    {
        /* Reset peripheral */
        __DAL_RCM_SPI2_FORCE_RESET();
        __DAL_RCM_SPI2_RELEASE_RESET();
        
        /* Configures SPI GPIO */
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_12 | GPIO_PIN_13 | GPIO_PIN_15);
    }

    if(hi2s->Instance == SPI3)
    {
        /* Reset peripheral */
        __DAL_RCM_SPI3_FORCE_RESET();
        __DAL_RCM_SPI3_RELEASE_RESET();
        
        /* Configures SPI GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_15);
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_3 | GPIO_PIN_5);
    }
}

/**
 * @brief   I2S configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_I2S_Config(void)
{
    /* Configures I2S2 */
    hi2s2.Instance              = SPI2;
    hi2s2.Init.Mode             = I2S_MODE_SLAVE_RX;
    hi2s2.Init.Standard         = I2S_STANDARD_PHILIPS;
    hi2s2.Init.DataFormat       = I2S_DATAFORMAT_16B;
    hi2s2.Init.MCLKOutput       = I2S_MCLKOUTPUT_DISABLE;
    hi2s2.Init.AudioFreq        = I2S_AUDIOFREQ_48K;
    hi2s2.Init.CPOL             = I2S_CPOL_LOW;
    hi2s2.Init.ClockSource      = I2S_CLOCK_PLL;
    hi2s2.Init.FullDuplexMode   = I2S_FULLDUPLEXMODE_DISABLE;
    if (DAL_I2S_Init(&hi2s2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configures I2S3 */
    hi2s3.Instance              = SPI3;
    hi2s3.Init.Mode             = I2S_MODE_MASTER_TX;
    hi2s3.Init.Standard         = I2S_STANDARD_PHILIPS;
    hi2s3.Init.DataFormat       = I2S_DATAFORMAT_16B;
    hi2s3.Init.MCLKOutput       = I2S_MCLKOUTPUT_DISABLE;
    hi2s3.Init.AudioFreq        = I2S_AUDIOFREQ_48K;
    hi2s3.Init.CPOL             = I2S_CPOL_LOW;
    hi2s3.Init.ClockSource      = I2S_CLOCK_PLL;
    hi2s3.Init.FullDuplexMode   = I2S_FULLDUPLEXMODE_DISABLE;
    if (DAL_I2S_Init(&hi2s3) != DAL_OK)
    {
        Error_Handler();
    }
}
