/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";

/* Private function prototypes ********************************************/
static void PMU_BOR_Config(uint8_t level);

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Check BOR reset flag */
    if (__DAL_RCM_GET_FLAG(RCM_FLAG_BORRST) != RESET)
    {
        LOGE(tag, "BOR Reset\r\n");
        __DAL_RCM_CLEAR_RESET_FLAGS();
    }

    /* Infinite loop */
    while (1)
    {
        if (!BOARD_BUTTON_GetState(BUTTON_KEY1))
        {
            DAL_Delay(10U);
            if (!BOARD_BUTTON_GetState(BUTTON_KEY1))
            {
                PMU_BOR_Config(OB_BOR_LEVEL3);
            }
            while (!BOARD_BUTTON_GetState(BUTTON_KEY1))
            {
            }
        }
    }
}

/**
 * @brief   Configure Brown-out Reset
 *  
 * @param   level :BOR level
 *            This parameter can be a value of @ref FLASHEx_BOR_Reset_Level 
 *            @arg OB_BOR_LEVEL3 : Supply voltage ranges from 2.7 to 3.6 V
 *            @arg OB_BOR_LEVEL2 : Supply voltage ranges from 2.4 to 2.7 V
 *            @arg OB_BOR_LEVEL1 : Supply voltage ranges from 2.1 to 2.4 V
 *            @arg OB_BOR_OFF : Supply voltage ranges from 1.62 to 2.1 V
 *  
 * @retval  None
 */
static void PMU_BOR_Config(uint8_t level)
{
    FLASH_OBProgramInitTypeDef OB_InitStruct = {0};
    
    DAL_FLASHEx_OBGetConfig(&OB_InitStruct);
    
    /* Get BOR Option Bytes */
    if ((OB_InitStruct.BORLevel & 0x0C) != level)
    {
        /* Unlocks the option bytes */
        DAL_FLASH_OB_Unlock();

        /* Select the desired V(BOR) Level */
        OB_InitStruct.OptionType = OPTIONBYTE_BOR;
        OB_InitStruct.BORLevel = level;
        DAL_FLASHEx_OBProgram(&OB_InitStruct);

        /* Launch the option byte */
        DAL_FLASH_OB_Launch();

        /* Locks the option bytes */
        DAL_FLASH_OB_Lock();
        
        LOGI(tag, "The BOR value is set from 0x%02X to 0x%02X\r\n",OB_InitStruct.BORLevel,level);
    }
    else
    {
        LOGE(tag, "The BOR setting is the same as in option bytes\r\n");
    }
}
