/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2023-07-31
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/
#define TMR_PERIOD      0xFFFFU

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
uint16_t CC1_Val;
uint16_t CC2_Val;

TMR_HandleTypeDef htmr9;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR Output Compare MSP
 *
 * @param  htmr TMR Output Compare handle
 *
 * @retval None
 */
void DAL_TMR_OC_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR9)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR9_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF3_TMR9;

        /* TMR channel 1 */
        GPIO_InitStruct.Pin         = GPIO_PIN_2;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* TMR channel 2 */
        GPIO_InitStruct.Pin         = GPIO_PIN_3;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }
}

/**
 * @brief  DeInitializes TMR Output Compare MSP
 *
 * @param  htmr TMR Output Compare handle
 *
 * @retval None
 */
void DAL_TMR_OC_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR9)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR9_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_2 | GPIO_PIN_3);
    }
}

/**
 * @brief   TMR9 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR9_Config(void)
{
    CC1_Val = (SystemCoreClock / 10000U);
    CC2_Val = (SystemCoreClock / 100000U);

    TMR_OC_InitTypeDef OC_ConfigStruct = {0};

    htmr9.Instance                  = TMR9;
    htmr9.Init.Prescaler            = 0U;
    htmr9.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr9.Init.Period               = TMR_PERIOD - 1U;
    htmr9.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr9.Init.RepetitionCounter    = 0U;
    htmr9.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_OC_Init(&htmr9) != DAL_OK)
    {
        Error_Handler();
    }
    
    /* Configure channel */
    OC_ConfigStruct.OCMode          = TMR_OCMODE_TOGGLE;
    OC_ConfigStruct.OCPolarity      = TMR_OCPOLARITY_LOW;

    /* Set the pulse value for channel 1 */
    OC_ConfigStruct.Pulse           = CC1_Val;
    if (DAL_TMR_OC_ConfigChannel(&htmr9, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 2 */
    OC_ConfigStruct.Pulse           = CC2_Val;
    if (DAL_TMR_OC_ConfigChannel(&htmr9, &OC_ConfigStruct, TMR_CHANNEL_2) != DAL_OK)
    {
        Error_Handler();
    }
}
