/**
 * @file        apm32f4xx_cryp_cfg.c
 *
 * @brief       This file provides configuration support for CRYP
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_cryp_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
CRYP_HandleTypeDef hcryp;

/* AES keys */
uint32_t AES128Key[4] =
{
    0x2B7E1516U ,0x28AED2A6U ,0xABF71588U ,0x09CF4F3CU
};

uint32_t AES192Key[6] =
{
    0x8E73B0F7U ,0xDA0E6452U ,0xC810F32BU ,0x809079E5U ,0x62F8EAD2U ,0x522C6B7BU
};

uint32_t AES256Key[8] =
{
    0x603DEB10U ,0x15CA71BEU ,0x2B73AEF0U ,0x857D7781U ,0x1F352C07U ,0x3B6108D7U ,0x2D9810A3U ,0x0914DFF4U
};

/* Initialization vector */
uint32_t AESIV_CBC[4] =
{
    0x00010203U, 0x04050607U, 0x08090A0BU, 0x0C0D0E0FU
};

uint32_t AESIV_CTR[4] =
{
    0xF0F1F2F3U, 0xF4F5F6F7U, 0xF8F9FAFBU, 0xFCFDFEFFU
};

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the CRYP MSP
 *
 * @param  hcryp: pointer to a CRYP_HandleTypeDef structure that contains
 *         the configuration information for CRYP module
 *
 * @retval None
 */
void DAL_CRYP_MspInit(CRYP_HandleTypeDef *hcryp)
{
    /* Enable peripheral clock */
    __DAL_RCM_CRYP_CLK_ENABLE();
}

/**
 * @brief  DeInitializes CRYP MSP
 (
 * @param  hcryp: pointer to a CRYP_HandleTypeDef structure that contains
 *         the configuration information for CRYP module
 *
 * @retval None
 */
void DAL_CRYP_MspDeInit(CRYP_HandleTypeDef *hcryp)
{
    /* Disable peripheral clock */
    __DAL_RCM_CRYP_CLK_DISABLE();
}

/**
 * @brief   CRYP AES 128 ECB configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_CRYP_AESECBConfig(void)
{
    hcryp.Instance              = CRYP;
    hcryp.Init.DataType         = CRYP_DATATYPE_32B;
    hcryp.Init.KeySize          = CRYP_KEYSIZE_128B;
    hcryp.Init.pKey             = AES128Key;
    hcryp.Init.Algorithm        = CRYP_AES_ECB;
    hcryp.Init.KeyIVConfigSkip  = CRYP_KEYIVCONFIG_ALWAYS;
    if (DAL_CRYP_Init(&hcryp) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
 * @brief   CRYP AES 192 CBC configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_CRYP_AESCBCConfig(void)
{
    hcryp.Instance              = CRYP;
    hcryp.Init.DataType         = CRYP_DATATYPE_32B;
    hcryp.Init.KeySize          = CRYP_KEYSIZE_192B;
    hcryp.Init.pKey             = AES192Key;
    hcryp.Init.Algorithm        = CRYP_AES_CBC;
    hcryp.Init.pInitVect        = AESIV_CBC;
    if (DAL_CRYP_Init(&hcryp) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
 * @brief   CRYP AES 256 CTR configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_CRYP_AESCTRConfig(void)
{
    hcryp.Instance              = CRYP;
    hcryp.Init.DataType         = CRYP_DATATYPE_32B;
    hcryp.Init.KeySize          = CRYP_KEYSIZE_256B;
    hcryp.Init.pKey             = AES256Key;
    hcryp.Init.Algorithm        = CRYP_AES_CTR;
    hcryp.Init.pInitVect        = AESIV_CTR;
    if (DAL_CRYP_Init(&hcryp) != DAL_OK)
    {
        Error_Handler();
    }
}
