/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";

/* Private function prototypes ********************************************/
static void RTC_AlarmConfig(void);
static void RTC_ShowTime(void);

/* External variables *****************************************************/
extern RTC_HandleTypeDef hrtc;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Config alarm */
    RTC_AlarmConfig();
    
    /* Infinite loop */
    while (1)
    {
        RTC_ShowTime();
    }
}

/**
 * @brief   Alarm A callback
 *
 * @param   hrtc pointer to a RTC_HandleTypeDef structure that contains
 *                the configuration information for RTC
 *
 * @retval  None
 */
void DAL_RTC_AlarmAEventCallback(RTC_HandleTypeDef *hrtc)
{
    LOGE(tag, "RTC Alarm A Event\r\n");
}

/* Private functions ******************************************************/

/**
 * @brief   Display the current time
 * @param   None
 * @retval  None
 */
static void RTC_ShowTime(void)
{
    RTC_DateTypeDef Date_ConfigStruct = {0};
    RTC_TimeTypeDef Time_ConfigStruct = {0};
    
    static uint32_t temp = 0U;
    
    /* Get the RTC current Time */
    DAL_RTC_GetTime(&hrtc, &Time_ConfigStruct, RTC_FORMAT_BIN);
    /* Get the RTC current Date */
    DAL_RTC_GetDate(&hrtc, &Date_ConfigStruct, RTC_FORMAT_BIN);
    
    if (temp != Time_ConfigStruct.Seconds)
    {
        /* Display time format : hh:mm:ss */
        LOGI(tag,"Current Time:%02d:%02d:%02d\r\n",Time_ConfigStruct.Hours, Time_ConfigStruct.Minutes, Time_ConfigStruct.Seconds);
    }
    
    temp = Time_ConfigStruct.Seconds;
} 

/**
 * @brief   Configure the current time and date
 *
 * @param   None
 *
 * @retval  None
 */
static void RTC_AlarmConfig(void)
{
    RTC_DateTypeDef  Date_ConfigStruct = {0};
    RTC_TimeTypeDef  Time_ConfigStruct = {0};
    RTC_AlarmTypeDef Alarm_ConfigStruct = {0};

    /* Configure the RTC alarm peripheral */
    /* Set alarm to 05:20:10 */
    Alarm_ConfigStruct.Alarm                = RTC_ALARM_A;
    Alarm_ConfigStruct.AlarmDateWeekDay     = RTC_WEEKDAY_MONDAY;
    Alarm_ConfigStruct.AlarmDateWeekDaySel  = RTC_ALARMDATEWEEKDAYSEL_DATE;
    Alarm_ConfigStruct.AlarmMask            = RTC_ALARMMASK_DATEWEEKDAY;
    Alarm_ConfigStruct.AlarmSubSecondMask   = RTC_ALARMSUBSECONDMASK_NONE;
    Alarm_ConfigStruct.AlarmTime.TimeFormat = RTC_HOURFORMAT12_AM;
    Alarm_ConfigStruct.AlarmTime.Hours      = 0x05U;
    Alarm_ConfigStruct.AlarmTime.Minutes    = 0x20U;
    Alarm_ConfigStruct.AlarmTime.Seconds    = 0x10U;
    Alarm_ConfigStruct.AlarmTime.SubSeconds = 0x0FU;
    if (DAL_RTC_SetAlarm_IT(&hrtc,&Alarm_ConfigStruct,RTC_FORMAT_BCD) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure the date */
    /* Set date: Tuesday Tuly 31th 2023 */
    Date_ConfigStruct.Year      = 0x23U;
    Date_ConfigStruct.Month     = RTC_MONTH_JULY;
    Date_ConfigStruct.Date      = 0x31U;
    Date_ConfigStruct.WeekDay   = RTC_WEEKDAY_MONDAY;
    if (DAL_RTC_SetDate(&hrtc,&Date_ConfigStruct,RTC_FORMAT_BCD) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure the time */
    /* Set time: 05:20:01 */
    Time_ConfigStruct.Hours             = 0x05U;
    Time_ConfigStruct.Minutes           = 0x20U;
    Time_ConfigStruct.Seconds           = 0x01U;
    Time_ConfigStruct.TimeFormat        = RTC_HOURFORMAT12_AM;
    Time_ConfigStruct.DayLightSaving    = RTC_DAYLIGHTSAVING_NONE;
    Time_ConfigStruct.StoreOperation    = RTC_STOREOPERATION_RESET;
    if (DAL_RTC_SetTime(&hrtc,&Time_ConfigStruct,RTC_FORMAT_BCD) != DAL_OK)
    {
        Error_Handler();
    }
}
