/**
 * @file        apm32f4xx_adc_cfg.c
 *
 * @brief       This file provides configuration support for ADC
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_adc_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
ADC_HandleTypeDef hadc1;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the ADC MSP
 *
 * @param  hadc pointer to a ADC_HandleTypeDef structure that contains
 *         the configuration information for the specified ADC. 
 *
 * @retval None
 */
void DAL_ADC_MspInit(ADC_HandleTypeDef* hadc)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    if(hadc->Instance == ADC1)
    {
        /* Enable peripheral clock */
        __DAL_RCM_ADC1_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();

        /* ADC1 pin configuration */
        GPIO_InitStruct.Pin     = GPIO_PIN_0;
        GPIO_InitStruct.Mode    = GPIO_MODE_ANALOG;
        GPIO_InitStruct.Pull    = GPIO_NOPULL;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }
}

/**
 * @brief  DeInitializes the ADC MSP
 *
 * @param  hadc pointer to a ADC_HandleTypeDef structure that contains
 *         the configuration information for the specified ADC
 *
 * @retval None
 */
void DAL_ADC_MspDeInit(ADC_HandleTypeDef* hadc)
{
    if(hadc->Instance == ADC1)
    {
        /* Disable peripheral clock */
        __DAL_RCM_ADC1_CLK_DISABLE();

        /* ADC GPIO configuration */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_0);

        /* Disable peripheral interrupt */
        DAL_NVIC_DisableIRQ(ADC_IRQn);
    }
}

/**
 * @brief   ADC1 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_ADC1_Config(void)
{
    ADC_AnalogWDGConfTypeDef AWDG_ConfigStruct = {0};
    ADC_ChannelConfTypeDef Channel_ConfigStruct = {0};

    /* Configure the ADC1 */
    hadc1.Instance                      = ADC1;
    hadc1.Init.ClockPrescaler           = ADC_CLOCK_SYNC_PCLK_DIV4;
    hadc1.Init.Resolution               = ADC_RESOLUTION_12B;
    hadc1.Init.ScanConvMode             = DISABLE;
    hadc1.Init.ContinuousConvMode       = ENABLE;
    hadc1.Init.DiscontinuousConvMode    = DISABLE;
    hadc1.Init.ExternalTrigConvEdge     = ADC_EXTERNALTRIGCONVEDGE_NONE;
    hadc1.Init.ExternalTrigConv         = ADC_SOFTWARE_START;
    hadc1.Init.DataAlign                = ADC_DATAALIGN_RIGHT;
    hadc1.Init.NbrOfConversion          = 1U;
    hadc1.Init.DMAContinuousRequests    = DISABLE;
    hadc1.Init.EOCSelection             = ADC_EOC_SINGLE_CONV;
    if (DAL_ADC_Init(&hadc1) != DAL_OK)
    {
        Error_Handler();
    }
    
    /* Configure the analog watchdog */
    AWDG_ConfigStruct.WatchdogMode       = ADC_ANALOGWATCHDOG_SINGLE_REG;
    AWDG_ConfigStruct.HighThreshold      = 0xB00U;
    AWDG_ConfigStruct.LowThreshold       = 0x300U;
    AWDG_ConfigStruct.Channel            = ADC_CHANNEL_0;
    AWDG_ConfigStruct.ITMode             = ENABLE;
    if (DAL_ADC_AnalogWDGConfig(&hadc1, &AWDG_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
    
    /* Configure for the selected ADC regular channel */
    Channel_ConfigStruct.Channel        = ADC_CHANNEL_0;
    Channel_ConfigStruct.Rank           = 1U;
    Channel_ConfigStruct.SamplingTime   = ADC_SAMPLETIME_3CYCLES;
    if (DAL_ADC_ConfigChannel(&hadc1, &Channel_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}
