/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"
#include <string.h>
#include <stdbool.h>

/* Private macro **********************************************************/

/* Define firmware to switch between Master board or Slave board */
#define IS_MASTER_BOARD

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";
static char txBuffer[] = "I2C_TwoBoardsPolling\r\n";
static char rxBuffer[50U];

/* Private function prototypes ********************************************/
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size);

/* External variables *****************************************************/
extern I2C_HandleTypeDef hi2c1;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

#if defined(IS_MASTER_BOARD)
    while (DAL_I2C_Master_Transmit(&hi2c1, (uint16_t)I2C_SALVE_ADDRESS, (uint8_t*)txBuffer, strlen(txBuffer), 10000U)!= DAL_OK)
    {
        if (DAL_I2C_GetError(&hi2c1) != DAL_I2C_ERROR_AF)
        {
            Error_Handler();
        }
    }

    while (DAL_I2C_Master_Receive(&hi2c1, (uint16_t)I2C_SALVE_ADDRESS, (uint8_t *)rxBuffer, strlen(txBuffer), 10000U) != DAL_OK)
    {
        if (DAL_I2C_GetError(&hi2c1) != DAL_I2C_ERROR_AF)
        {
            Error_Handler();
        }
    }
#else
    if (DAL_I2C_Slave_Receive(&hi2c1, (uint8_t *)rxBuffer, strlen(txBuffer), 10000U) != DAL_OK)
    {
        Error_Handler();
    }

    if (DAL_I2C_Slave_Transmit(&hi2c1, (uint8_t*)txBuffer, strlen(txBuffer), 10000U)!= DAL_OK)
    {
        Error_Handler();
    }
#endif

    if (BufferCmp((uint8_t*)txBuffer, (uint8_t*)rxBuffer, strlen(txBuffer)) != true)
    {
        BOARD_LED_On(LED3);
        LOGE(tag, "I2C Polling Communication Error\r\n");
        Error_Handler();
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500U);
    }
}

/**
 * @brief   Buffer compare
 *
 * @param   buf1: First buffer to be compared
 *
 * @param   buf2: Second buffer to be compared
 *
 * @param   size: Buffer size
 *
 * @retval  Comparison result(ture or false)
 */
static bool BufferCmp(uint8_t *buf1, uint8_t *buf2, uint32_t size)
{
    uint32_t i;

    bool status = true;

    for (i = 0U; i < size; i++)
    {
        if (buf1[i] != buf2[i])
        {
            status = false;
            break;
        }
    }

    return status;
}
