/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"
#include "cmsis_os2.h"

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/

static const char* tag = "main";

const osThreadAttr_t ThreadStart_Attr =
{
    .name = "osRtxStartThread",
    .attr_bits = osThreadDetached,
    .priority = osPriorityHigh4,
    .stack_size = 2048,
};

const osThreadAttr_t ThreadLedToggle_Attr =
{
    .name = "osRtxLedToggleThread",
    .attr_bits = osThreadDetached,
    .priority = osPriorityHigh2,
    .stack_size = 512,
};

const osThreadAttr_t ThreadUsart_Attr =
{
    .name = "osRtxUsartThread",
    .attr_bits = osThreadDetached,
    .priority = osPriorityHigh1,
    .stack_size = 1024,
};

/* Task handle */
osThreadId_t ThreadIdStart = NULL;
osThreadId_t ThreadIdTaskUsart = NULL;
osThreadId_t ThreadIdTaskLed = NULL;

/* Private function prototypes ********************************************/

/* User create task */
static void createThread(void);

/* Start task */
void startThread(void *argument);

/* Led toggle task */
void ledThread(void* argument);

/* Usart test task */
void usartThread(void* argument);

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_Init();

    DAL_SysClkConfig();
    
    /* Time base for closing DAL before kernel startup */
    DAL_SuspendTick();
    
    /* Initialize the RTOS Kernel */
    osKernelInitialize();

    /* Create a thread and add it to Active Threads */
    ThreadIdStart = osThreadNew(startThread, NULL, &ThreadStart_Attr);

    /* Start the RTOS Kernel scheduler */
    osKernelStart();

    while (1);
}

/**
 * @brief   The launch thread
 *
 * @param   argument
 *
 * @retval  None
 */
void startThread(void *argument)
{
    /* Delay period */
    const uint16_t usFrequency = 1;
    uint32_t tick;

    /* Configure peripheral */
    DAL_ResumeTick();

    BOARD_LED_Config(LED2);
    BOARD_LED_Config(LED3);

    DAL_USART1_Config();

    /* User task create */
    createThread();

    /* Get the RTOS kernel tick count */
    tick = osKernelGetTickCount();

    while (1)
    {
        tick += usFrequency;
        osDelayUntil(tick);
    }
}

/**
 * @brief   LED toggle task
 *
 * @param   argument
 *
 * @retval  None
 */
void ledThread(void* argument)
{
    /* Delay period */
    const uint16_t usFrequency = 500;
    uint32_t tick;

    /* Get the RTOS kernel tick count */
    tick = osKernelGetTickCount();

    while (1)
    {
        BOARD_LED_Toggle(LED2);
        tick += usFrequency;
        osDelayUntil(tick);
    }
}

/**
 * @brief   UART test task
 *
 * @param   argument
 *
 * @retval  None
 */
void usartThread(void* argument)
{
    /* Delay period */
    const uint16_t usFrequency = 1000;
    uint32_t tick;

    /* Get the RTOS kernel tick count */
    tick = osKernelGetTickCount();

    while (1)
    {
        LOGI(tag, "RTX5 Hello World!\r\n");
        tick += usFrequency;
        osDelayUntil(tick);
    }
}

/* Private functions ******************************************************/

/**
 * @brief   Create application thread
 *
 * @param   None
 *
 * @retval  None
 */
static void createThread(void)
{
    ThreadIdTaskLed = osThreadNew(ledThread, NULL, &ThreadLedToggle_Attr);
    ThreadIdTaskUsart = osThreadNew(usartThread, NULL, &ThreadUsart_Attr);
}
