/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2023-12-01
 *
 * @attention
 *
 *  Copyright (C) 2023 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/
#define TMR1_PERIOD     2048U

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr1;
TMR_OC_InitTypeDef OC_ConfigStruct = {0};

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR Output Compare MSP
 *
 * @param  htmr TMR Output Compare handle
 *
 * @retval None
 */
void DAL_TMR_OC_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR1)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR1_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();
        __DAL_RCM_GPIOB_CLK_ENABLE();
        __DAL_RCM_GPIOE_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLDOWN;
        GPIO_InitStruct.Speed       = GPIO_SPEED_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF1_TMR1;

        /* TMR channel 1 */
        GPIO_InitStruct.Pin         = GPIO_PIN_8;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* TMR channel 1N */
        GPIO_InitStruct.Pin         = GPIO_PIN_7;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);

        /* TMR channel 2 */
        GPIO_InitStruct.Pin         = GPIO_PIN_11;
        DAL_GPIO_Init(GPIOE, &GPIO_InitStruct);

        /* TMR channel 2N */
        GPIO_InitStruct.Pin         = GPIO_PIN_14;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        /* TMR channel 3 */
        GPIO_InitStruct.Pin         = GPIO_PIN_13;
        DAL_GPIO_Init(GPIOE, &GPIO_InitStruct);

        /* TMR channel 3N */
        GPIO_InitStruct.Pin         = GPIO_PIN_15;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        /* TMR break */
        GPIO_InitStruct.Pin         = GPIO_PIN_12;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
    }
}

/**
 * @brief  DeInitializes TMR Output Compare MSP
 *
 * @param  htmr TMR Output Compare handle
 *
 * @retval None
 */
void DAL_TMR_OC_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR1)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR1_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_7 | GPIO_PIN_8);
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_12 | GPIO_PIN_14 | GPIO_PIN_15);
        DAL_GPIO_DeInit(GPIOE, GPIO_PIN_11 | GPIO_PIN_13);

        /* Disable peripheral interrupt */
        DAL_NVIC_DisableIRQ(TMR1_UP_TMR10_IRQn);
    }
}

/**
 * @brief   TMR1 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR1_Config(void)
{
    TMR_BreakDeadTimeConfigTypeDef BKDT_ConfigStruct = {0};

    htmr1.Instance                  = TMR1;
    htmr1.Init.Prescaler            = 0U;
    htmr1.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr1.Init.Period               = TMR1_PERIOD - 1U;
    htmr1.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr1.Init.RepetitionCounter    = 0U;
    htmr1.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_OC_Init(&htmr1) != DAL_OK)
    {
        Error_Handler();
    }
    
    /* Configure channel */
    OC_ConfigStruct.OCMode          = TMR_OCMODE_TIMING;
    OC_ConfigStruct.OCPolarity      = TMR_OCPOLARITY_HIGH;
    OC_ConfigStruct.OCNPolarity     = TMR_OCNPOLARITY_HIGH;
    OC_ConfigStruct.OCIdleState     = TMR_OCIDLESTATE_SET;
    OC_ConfigStruct.OCNIdleState    = TMR_OCNIDLESTATE_SET;
    OC_ConfigStruct.OCFastMode      = TMR_OCFAST_DISABLE;

    /* Set the pulse value for channel 1 */
    OC_ConfigStruct.Pulse           = (TMR1_PERIOD / 2U) - 1U;
    if (DAL_TMR_OC_ConfigChannel(&htmr1, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 2 */
    OC_ConfigStruct.Pulse           = (TMR1_PERIOD / 4U) - 1U;
    if (DAL_TMR_OC_ConfigChannel(&htmr1, &OC_ConfigStruct, TMR_CHANNEL_2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 3 */
    OC_ConfigStruct.Pulse           = (TMR1_PERIOD / 8U) - 1U;
    if (DAL_TMR_OC_ConfigChannel(&htmr1, &OC_ConfigStruct, TMR_CHANNEL_3) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure break feature */
    BKDT_ConfigStruct.DeadTime          = 1U;
    BKDT_ConfigStruct.OffStateRunMode   = TMR_OSSR_ENABLE;
    BKDT_ConfigStruct.OffStateIDLEMode  = TMR_OSSI_ENABLE;
    BKDT_ConfigStruct.LockLevel         = TMR_LOCKLEVEL_OFF;
    BKDT_ConfigStruct.BreakState        = TMR_BREAK_ENABLE;
    BKDT_ConfigStruct.BreakPolarity     = TMR_BREAKPOLARITY_HIGH;
    BKDT_ConfigStruct.AutomaticOutput   = TMR_AUTOMATICOUTPUT_ENABLE;
    if (DAL_TMREx_ConfigBreakDeadTime(&htmr1, &BKDT_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
    
    /* Configure the commutation event */
    DAL_TMREx_ConfigCommutEvent_IT(&htmr1, TMR_TS_NONE, TMR_COMMUTATION_SOFTWARE);
}
