/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/
#include "apm32f4xx_device_cfg.h"
#include "log.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
static const char* tag = "main";
__IO uint32_t wakeupFlag = 0;

/* Private function prototypes ********************************************/
static void PMU_StopMode(void);

/* External variables *****************************************************/
extern RTC_HandleTypeDef hrtc;
extern UART_HandleTypeDef huart1;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    /* Device configuration */
    DAL_DeviceConfig();

    /* Enable PMU clock */
    __DAL_RCM_PMU_CLK_ENABLE();

    /* Wait until USER button is pressed to enter the Low Power mode */
    while (BOARD_BUTTON_GetState(BUTTON_KEY2) == GPIO_PIN_RESET)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(100);
    }

    /* Loop while USER Button is maintained pressed */
    while (BOARD_BUTTON_GetState(BUTTON_KEY2) != GPIO_PIN_RESET)
    {
    }

    PMU_StopMode();

    if (wakeupFlag != 0)
    {
        DAL_USART1_Config();
        BOARD_LED_Config(LED2);
        LOGI(tag, "Exit stop mode\r\n");
    }

    /* Infinite loop */
    while (1)
    {
        BOARD_LED_Toggle(LED2);
        DAL_Delay(500);
    }
}

/**
 * @brief   PMU stop mode
 *  
 * @param   None
 *  
 * @retval  None
 */
static void PMU_StopMode(void)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};
    RCM_ClkInitTypeDef RCM_ClkInitStruct = {0};
    RCM_OscInitTypeDef RCM_OscInitStruct = {0};
    uint32_t flashLatency = 0U;

    LOGI(tag, "Enter stop mode\r\n");

    /* Configure all GPIO as analog to reduce current consumption */
    /* Enable GPIOs clock */
    __DAL_RCM_GPIOA_CLK_ENABLE();
    __DAL_RCM_GPIOB_CLK_ENABLE();
    __DAL_RCM_GPIOC_CLK_ENABLE();
    __DAL_RCM_GPIOD_CLK_ENABLE();
    __DAL_RCM_GPIOE_CLK_ENABLE();
    __DAL_RCM_GPIOF_CLK_ENABLE();
    __DAL_RCM_GPIOG_CLK_ENABLE();
    __DAL_RCM_GPIOH_CLK_ENABLE();

    GPIO_InitStruct.Pin     = GPIO_PIN_All;
    GPIO_InitStruct.Mode    = GPIO_MODE_ANALOG;
    GPIO_InitStruct.Speed   = GPIO_SPEED_LOW;
    GPIO_InitStruct.Pull    = GPIO_NOPULL;

    DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
    DAL_GPIO_Init(GPIOC, &GPIO_InitStruct);
    DAL_GPIO_Init(GPIOD, &GPIO_InitStruct);
    DAL_GPIO_Init(GPIOE, &GPIO_InitStruct);
    DAL_GPIO_Init(GPIOF, &GPIO_InitStruct);
    DAL_GPIO_Init(GPIOG, &GPIO_InitStruct);
    DAL_GPIO_Init(GPIOH, &GPIO_InitStruct);

    /* Disable GPIOs clock */
    __DAL_RCM_GPIOA_CLK_DISABLE();
    __DAL_RCM_GPIOB_CLK_DISABLE();
    __DAL_RCM_GPIOC_CLK_DISABLE();
    __DAL_RCM_GPIOD_CLK_DISABLE();
    __DAL_RCM_GPIOE_CLK_DISABLE();
    __DAL_RCM_GPIOF_CLK_DISABLE();
    __DAL_RCM_GPIOG_CLK_DISABLE();
    __DAL_RCM_GPIOH_CLK_DISABLE();

    /* Reset UART state for configure again and deinit UART */
    DAL_UART_DeInit(&huart1);

    /* RTC Config */
    DAL_RTC_Config();

    /* Sets wake-up time = 10s */
    DAL_RTCEx_SetWakeUpTimer_IT(&hrtc, 0x4E20U, RTC_WAKEUPCLOCK_RTCCLK_DIV16);

    /* FLASH deep power down mode enabled */
    DAL_PMUEx_EnableFlashPowerDown();

    /* Enter stop */
    DAL_PMU_EnterSTOPMode(PMU_LOWPOWERREGULATOR_ON, PMU_STOPENTRY_WFI);
    
    /* Configures system clock after wake-up from STOP mode */
    /* Get the oscillators configuration according to the internal RCM registers */
    DAL_RCM_GetOscConfig(&RCM_OscInitStruct);

    /* Configures oscillators */
    RCM_OscInitStruct.OscillatorType    = RCM_OSCILLATORTYPE_HSE;
    RCM_OscInitStruct.HSEState          = RCM_HSE_ON;
    RCM_OscInitStruct.PLL.PLLState      = RCM_PLL_ON;
    if (DAL_RCM_OscConfig(&RCM_OscInitStruct) != DAL_OK)
    {
        Error_Handler();
    }

    /* Get the clocks configuration according to the internal RCM registers */
    DAL_RCM_GetClockConfig(&RCM_ClkInitStruct, &flashLatency);
    
    /* Configure the HCLK, PCLK1 and PCLK2 clocks dividers */
    RCM_ClkInitStruct.ClockType     = RCM_CLOCKTYPE_SYSCLK;
    RCM_ClkInitStruct.SYSCLKSource  = RCM_SYSCLKSOURCE_PLLCLK;
    if (DAL_RCM_ClockConfig(&RCM_ClkInitStruct, flashLatency) != DAL_OK)
    {
        Error_Handler();
    }

    /* Disable Wake-up timer */
    if(DAL_RTCEx_DeactivateWakeUpTimer(&hrtc) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
  * @brief  Wakeup Timer callback.
  * @param  hrtc pointer to a RTC_HandleTypeDef structure that contains
  *                the configuration information for RTC.
  * @retval None
  */
void DAL_RTCEx_WakeUpTimerEventCallback(RTC_HandleTypeDef *hrtc)
{
    wakeupFlag = 1;
}
