/**
 * @file        apm32f4xx_spi_cfg.c
 *
 * @brief       This file provides configuration support for SPI
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_spi_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
SPI_HandleTypeDef hspi1;
SPI_HandleTypeDef hspi2;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initialize the SPI MSP
 *
 * @param  hspi pointer to a SPI_HandleTypeDef structure that contains
 *               the configuration information for SPI module
 *
 * @retval None
 */
void DAL_SPI_MspInit(SPI_HandleTypeDef *hspi)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0U};

    if(hspi->Instance == SPI1)
    {
        /* Enable peripheral clock */
        __DAL_RCM_SPI1_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOA_CLK_ENABLE();
        __DAL_RCM_GPIOB_CLK_ENABLE();

        /* Configure the SPI pin */
        GPIO_InitStruct.Pin         = GPIO_PIN_3 | GPIO_PIN_4 | GPIO_PIN_5;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;
        GPIO_InitStruct.Alternate   = GPIO_AF5_SPI1;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);

        GPIO_InitStruct.Pin         = GPIO_PIN_15;
        DAL_GPIO_Init(GPIOA, &GPIO_InitStruct);
    }

    if(hspi->Instance == SPI2)
    {
        /* Enable peripheral clock */
        __DAL_RCM_SPI2_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOB_CLK_ENABLE();
        __DAL_RCM_GPIOC_CLK_ENABLE();

        /* Configure the SPI pin */
        GPIO_InitStruct.Pin         = GPIO_PIN_12 | GPIO_PIN_13;
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_FAST;
        GPIO_InitStruct.Alternate   = GPIO_AF5_SPI2;
        DAL_GPIO_Init(GPIOB, &GPIO_InitStruct);
        
        GPIO_InitStruct.Pin         = GPIO_PIN_2 | GPIO_PIN_3;
        DAL_GPIO_Init(GPIOC, &GPIO_InitStruct);
    }
}

/**
 * @brief  De-Initialize the SPI MSP
 *
 * @param  hspi pointer to a SPI_HandleTypeDef structure that contains
 *               the configuration information for SPI module
 *
 * @retval None
 */
void DAL_SPI_MspDeInit(SPI_HandleTypeDef *hspi)
{
    if(hspi->Instance == SPI1)
    {
        /* Reset peripheral */
        __DAL_RCM_SPI1_FORCE_RESET();
        __DAL_RCM_SPI1_RELEASE_RESET();

        /* Configures SPI GPIO */
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_3 | GPIO_PIN_4 | GPIO_PIN_5);
        DAL_GPIO_DeInit(GPIOA, GPIO_PIN_15);
    }

    if(hspi->Instance == SPI2)
    {
        /* Reset peripheral */
        __DAL_RCM_SPI2_FORCE_RESET();
        __DAL_RCM_SPI2_RELEASE_RESET();
        
        /* Configures SPI GPIO */
        DAL_GPIO_DeInit(GPIOB, GPIO_PIN_12 | GPIO_PIN_13);
        DAL_GPIO_DeInit(GPIOC, GPIO_PIN_2 | GPIO_PIN_3);
    }
}

/**
 * @brief   SPI configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_SPI_Config(void)
{
    /* Configures SPI1 */
    hspi1.Instance                  = SPI1;
    hspi1.Init.Mode                 = SPI_MODE_MASTER;
    hspi1.Init.Direction            = SPI_DIRECTION_2LINES;
    hspi1.Init.DataSize             = SPI_DATASIZE_8BIT;
    hspi1.Init.CLKPolarity          = SPI_POLARITY_HIGH;
    hspi1.Init.CLKPhase             = SPI_PHASE_1EDGE;
    hspi1.Init.NSS                  = SPI_NSS_SOFT;
    hspi1.Init.BaudRatePrescaler    = SPI_BAUDRATEPRESCALER_256;
    hspi1.Init.FirstBit             = SPI_FIRSTBIT_MSB;
    hspi1.Init.TIMode               = SPI_TIMODE_DISABLE;
    hspi1.Init.CRCCalculation       = SPI_CRCCALCULATION_DISABLE;
    hspi1.Init.CRCPolynomial        = 7U;
    if (DAL_SPI_Init(&hspi1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configures SPI2 */
    hspi2.Instance                  = SPI2;
    hspi2.Init.Mode                 = SPI_MODE_SLAVE;
    hspi2.Init.Direction            = SPI_DIRECTION_2LINES;
    hspi2.Init.DataSize             = SPI_DATASIZE_8BIT;
    hspi2.Init.CLKPolarity          = SPI_POLARITY_HIGH;
    hspi2.Init.CLKPhase             = SPI_PHASE_1EDGE;
    hspi2.Init.NSS                  = SPI_NSS_SOFT;
    hspi2.Init.BaudRatePrescaler    = SPI_BAUDRATEPRESCALER_256;
    hspi2.Init.FirstBit             = SPI_FIRSTBIT_MSB;
    hspi2.Init.TIMode               = SPI_TIMODE_DISABLE;
    hspi2.Init.CRCCalculation       = SPI_CRCCALCULATION_DISABLE;
    hspi2.Init.CRCPolynomial        = 7U;
    if (DAL_SPI_Init(&hspi2) != DAL_OK)
    {
        Error_Handler();
    }
}
