/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr9;
TMR_HandleTypeDef htmr10;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR Base MSP
 *
 * @param  htmr TMR Base handle
 *
 * @retval None
 */
void DAL_TMR_Base_MspInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR9)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR9_CLK_ENABLE();
    }

    if (htmr->Instance == TMR10)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR10_CLK_ENABLE();
    }
}

/**
 * @brief  DeInitializes TMR Base MSP
 *
 * @param  htmr TMR Base handle
 *
 * @retval None
 */
void DAL_TMR_Base_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR9)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR9_CLK_DISABLE();

        /* Disable peripheral interrupt */
        DAL_NVIC_DisableIRQ(TMR1_BRK_TMR9_IRQn);
    }

    if (htmr->Instance == TMR10)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR10_CLK_DISABLE();

        /* Disable peripheral interrupt */
        DAL_NVIC_DisableIRQ(TMR1_UP_TMR10_IRQn);
    }
}

/**
 * @brief   TMR9 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR9_Config(void)
{
    TMR_SlaveConfigTypeDef Slave_ConfigStruct = {0};

    htmr9.Instance                  = TMR9;
    htmr9.Init.Prescaler            = 0U;
    htmr9.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr9.Init.Period               = 0xFFFF;
    htmr9.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr9.Init.RepetitionCounter    = 0U;
    htmr9.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_ENABLE;
    if (DAL_TMR_Base_Init(&htmr9) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure slave TMR */
    Slave_ConfigStruct.SlaveMode     = TMR_SLAVEMODE_EXTERNAL1;
    Slave_ConfigStruct.InputTrigger  = TMR_TS_ITR3;
    if (DAL_TMR_SlaveConfigSynchro(&htmr9, &Slave_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}

/**
 * @brief   TMR10 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR10_Config(void)
{
    TMR_MasterConfigTypeDef Master_ConfigStruct = {0};

    htmr10.Instance                  = TMR10;
    htmr10.Init.Prescaler            = 167U;
    htmr10.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr10.Init.Period               = 0xFFFFU;
    htmr10.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr10.Init.RepetitionCounter    = 0U;
    htmr10.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_ENABLE;
    if (DAL_TMR_Base_Init(&htmr10) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure master TMR */
    Master_ConfigStruct.MasterOutputTrigger = TMR_TRGO_UPDATE;
    Master_ConfigStruct.MasterSlaveMode     = TMR_MASTERSLAVEMODE_ENABLE;
    if (DAL_TMREx_MasterConfigSynchronization(&htmr10,&Master_ConfigStruct) != DAL_OK)
    {
        Error_Handler();
    }
}
