/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/
#define TMR_PERIOD      0xFFFFU

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
uint16_t CC1_Val = (120000000U / 5000U);
uint16_t CC2_Val = (120000000U / 10000U);
uint16_t CC3_Val = (120000000U / 50000U);
uint16_t CC4_Val = (120000000U / 100000U);

TMR_HandleTypeDef htmr4;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR Output Compare MSP
 *
 * @param  htmr TMR Output Compare handle
 *
 * @retval None
 */
void DAL_TMR_OC_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR4)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR4_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOD_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF2_TMR4;

        /* TMR channel 1 */
        GPIO_InitStruct.Pin         = GPIO_PIN_12;
        DAL_GPIO_Init(GPIOD, &GPIO_InitStruct);

        /* TMR channel 2 */
        GPIO_InitStruct.Pin         = GPIO_PIN_13;
        DAL_GPIO_Init(GPIOD, &GPIO_InitStruct);

        /* TMR channel 3 */
        GPIO_InitStruct.Pin         = GPIO_PIN_14;
        DAL_GPIO_Init(GPIOD, &GPIO_InitStruct);

        /* TMR channel 4 */
        GPIO_InitStruct.Pin         = GPIO_PIN_15;
        DAL_GPIO_Init(GPIOD, &GPIO_InitStruct);
    }
}

/**
 * @brief  DeInitializes TMR Output Compare MSP
 *
 * @param  htmr TMR Output Compare handle
 *
 * @retval None
 */
void DAL_TMR_OC_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR4)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR4_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOD, GPIO_PIN_12 | GPIO_PIN_13 | \
                               GPIO_PIN_14 | GPIO_PIN_15);
    }
}

/**
 * @brief   TMR4 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR4_Config(void)
{
    TMR_OC_InitTypeDef OC_ConfigStruct = {0};

    htmr4.Instance                  = TMR4;
    htmr4.Init.Prescaler            = 0U;
    htmr4.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr4.Init.Period               = TMR_PERIOD - 1U;
    htmr4.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr4.Init.RepetitionCounter    = 0U;
    htmr4.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_OC_Init(&htmr4) != DAL_OK)
    {
        Error_Handler();
    }
    
    /* Configure channel */
    OC_ConfigStruct.OCMode          = TMR_OCMODE_TOGGLE;
    OC_ConfigStruct.OCPolarity      = TMR_OCPOLARITY_LOW;

    /* Set the pulse value for channel 1 */
    OC_ConfigStruct.Pulse           = CC1_Val;
    if (DAL_TMR_OC_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 2 */
    OC_ConfigStruct.Pulse           = CC2_Val;
    if (DAL_TMR_OC_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_2) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 3 */
    OC_ConfigStruct.Pulse           = CC3_Val;
    if (DAL_TMR_OC_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_3) != DAL_OK)
    {
        Error_Handler();
    }

    /* Set the pulse value for channel 4 */
    OC_ConfigStruct.Pulse           = CC4_Val;
    if (DAL_TMR_OC_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_4) != DAL_OK)
    {
        Error_Handler();
    }
}
