/**
 * @file        apm32f4xx_tmr_cfg.c
 *
 * @brief       This file provides configuration support for TMR
 *
 * @version     V1.0.0
 *
 * @date        2025-05-08
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "apm32f4xx_tmr_cfg.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
TMR_HandleTypeDef htmr4;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief  Initializes the TMR PWM MSP
 *
 * @param  htmr TMR PWM handle
 *
 * @retval None
 */
void DAL_TMR_PWM_MspInit(TMR_HandleTypeDef *htmr)
{
    GPIO_InitTypeDef GPIO_InitStruct = {0};

    if (htmr->Instance == TMR4)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR4_CLK_ENABLE();

        /* Enable peripheral GPIO clock */
        __DAL_RCM_GPIOD_CLK_ENABLE();

        /* TMR channel pin configuration */
        GPIO_InitStruct.Mode        = GPIO_MODE_AF_PP;
        GPIO_InitStruct.Pull        = GPIO_PULLUP;
        GPIO_InitStruct.Speed       = GPIO_SPEED_HIGH;
        GPIO_InitStruct.Alternate   = GPIO_AF2_TMR4;

        /* TMR channel 1  : PD12
           TMR channel 2  : PD13
           TMR channel 3  : PD14
           TMR channel 4  : PD15
        */
        GPIO_InitStruct.Pin         = GPIO_PIN_12 | GPIO_PIN_13 | \
                                      GPIO_PIN_14 | GPIO_PIN_15;
        DAL_GPIO_Init(GPIOD, &GPIO_InitStruct);
    }
}

/**
 * @brief  DeInitializes TMR PWM MSP
 *
 * @param  htmr TMR PWM handle
 *
 * @retval None
 */
void DAL_TMR_PWM_MspDeInit(TMR_HandleTypeDef *htmr)
{
    if (htmr->Instance == TMR4)
    {
        /* Enable peripheral clock */
        __DAL_RCM_TMR4_CLK_DISABLE();

        /* De-init peripheral GPIO */
        DAL_GPIO_DeInit(GPIOD, GPIO_PIN_12 | GPIO_PIN_13 | \
                               GPIO_PIN_14 | GPIO_PIN_14);
    }
}

/**
 * @brief   TMR4 configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DAL_TMR4_Config(void)
{
    TMR_OC_InitTypeDef OC_ConfigStruct = {0};

    htmr4.Instance                  = TMR4;
    htmr4.Init.Prescaler            = 119U;
    htmr4.Init.CounterMode          = TMR_COUNTERMODE_UP;
    htmr4.Init.Period               = 999U;
    htmr4.Init.ClockDivision        = TMR_CLOCKDIVISION_DIV1;
    htmr4.Init.RepetitionCounter    = 0U;
    htmr4.Init.AutoReloadPreload    = TMR_AUTORELOAD_PRELOAD_DISABLE;
    if (DAL_TMR_PWM_Init(&htmr4) != DAL_OK)
    {
        Error_Handler();
    }

    /* Configure output channel 1 */
    OC_ConfigStruct.OCMode          = TMR_OCMODE_PWM1;
    OC_ConfigStruct.OCPolarity      = TMR_OCPOLARITY_HIGH;
    OC_ConfigStruct.Pulse           = 800U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_1) != DAL_OK)
    {
        Error_Handler();
    }

    OC_ConfigStruct.Pulse           = 700U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_2) != DAL_OK)
    {
        Error_Handler();
    }

    OC_ConfigStruct.Pulse           = 600U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_3) != DAL_OK)
    {
        Error_Handler();
    }

    OC_ConfigStruct.Pulse           = 500U;
    if (DAL_TMR_PWM_ConfigChannel(&htmr4, &OC_ConfigStruct, TMR_CHANNEL_4) != DAL_OK)
    {
        Error_Handler();
    }
}
