/*!
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2025-02-15
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"
#include "Board.h"

/* Private includes *******************************************************/
#include "apm32f4xx_adc.h"
#include "apm32f4xx_gpio.h"
#include "apm32f4xx_misc.h"
#include "apm32f4xx_usart.h"
#include "apm32f4xx_rcm.h"
#include "board_delay.h"
#include "log_printf.h"
#include <stdio.h>

/* Private macro **********************************************************/
#define DEBUG_USART  USART1

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
/* DMA value from ADC Dual Interleaved Mode*/
uint32_t DMA_TripleConvertedValue[3] = {0};

/* ADC convert to voltage*/
float ADC1_ConvertedValue1 = 0;
float ADC2_ConvertedValue1 = 0;
float ADC3_ConvertedValue1 = 0;
float ADC1_ConvertedValue2 = 0;
float ADC2_ConvertedValue2 = 0;
float ADC3_ConvertedValue2 = 0;

static const char* tag = "main"; 

/* Private function prototypes ********************************************/
void ADC_Init(void);
/* DMA Init */
void DMA_Init(uint32_t* Buf);

/* External variables *****************************************************/

/* External functions *****************************************************/

/*!
 * @brief       Main program
 *
 * @param       None
 *
 * @retval      None
 */
int main(void)
{
    USART_Config_T usartConfigStruct;

    /* USART configuration */
    USART_ConfigStructInit(&usartConfigStruct);
    usartConfigStruct.baudRate = 115200;
    usartConfigStruct.mode = USART_MODE_TX_RX;
    usartConfigStruct.parity = USART_PARITY_NONE;
    usartConfigStruct.stopBits = USART_STOP_BIT_1;
    usartConfigStruct.wordLength = USART_WORD_LEN_8B;
    usartConfigStruct.hardwareFlow = USART_HARDWARE_FLOW_NONE;

    /* COM1 init*/
    BOARD_COMInit(COM1, &usartConfigStruct);

    /* Configure LED GPIO Pin */
    BOARD_LEDInit(LED2);

    /* Configure Delay */
    BOARD_DelayConfig();

    DMA_Init(DMA_TripleConvertedValue);
    /* ADC init */
    ADC_Init();

    while (1)
    {
        if (DMA_ReadStatusFlag(DMA2_Stream0, DMA_FLAG_TCIFLG0))
        {
            ADC1_ConvertedValue1 = (float)(DMA_TripleConvertedValue[0] & 0xFFFF)/4095*3.3f;
            ADC2_ConvertedValue1 = (float)(DMA_TripleConvertedValue[0] >>16)/4095*3.3f;

            ADC3_ConvertedValue1 = (float)(DMA_TripleConvertedValue[1] & 0xFFFF)/4095*3.3f;
            ADC1_ConvertedValue2 = (float)(DMA_TripleConvertedValue[1] >>16)/4095*3.3f;

            ADC2_ConvertedValue2 = (float)(DMA_TripleConvertedValue[2] & 0xFFFF)/4095*3.3f;
            ADC3_ConvertedValue2 = (float)(DMA_TripleConvertedValue[2] >>16)/4095*3.3f;

            LOGI(tag, "ADC CDATA = 0x%04X,0x%04X,0x%04X \n",
            (uint16_t)DMA_TripleConvertedValue[0],
            (uint16_t)DMA_TripleConvertedValue[1],
            (uint16_t)DMA_TripleConvertedValue[2]);

            LOGI(tag, "ADC CH0: ADC1 Voltage1    = %f V \n", ADC1_ConvertedValue1);
            LOGI(tag, "ADC CH0: ADC2 Voltage1    = %f V \n", ADC2_ConvertedValue1);
            LOGI(tag, "ADC CH0: ADC3 Voltage1    = %f V \n", ADC3_ConvertedValue1);

            LOGI(tag, "ADC CH0: ADC1 Voltage2    = %f V \n", ADC1_ConvertedValue2);
            LOGI(tag, "ADC CH0: ADC2 Voltage2    = %f V \n", ADC2_ConvertedValue2);
            LOGI(tag, "ADC CH0: ADC3 Voltage2    = %f V \n", ADC3_ConvertedValue2);

            BOARD_DelayMs(500);
            DMA_ClearStatusFlag(DMA2_Stream0, DMA_FLAG_TCIFLG0);
        }
    }
}

/*!
 * @brief     ADC Init
 *
 * @param     None
 *
 * @retval    None
 */
void ADC_Init(void)
{
    GPIO_Config_T gpioConfig;
    ADC_Config_T  adcConfig;
    ADC_CommonConfig_T adcCommonConfig;

    /* RCM Enable*/
    RCM_EnableAHB1PeriphClock(RCM_AHB1_PERIPH_GPIOA);
    /* GPIO Configuration */
    GPIO_ConfigStructInit(&gpioConfig);
    gpioConfig.pin = GPIO_PIN_0;
    gpioConfig.mode = GPIO_MODE_AN;
    gpioConfig.pupd = GPIO_PUPD_NOPULL;
    GPIO_Config(GPIOA, &gpioConfig);

    RCM_EnableAPB2PeriphClock(RCM_APB2_PERIPH_ADC1);
    RCM_EnableAPB2PeriphClock(RCM_APB2_PERIPH_ADC2);
    RCM_EnableAPB2PeriphClock(RCM_APB2_PERIPH_ADC3);

    ADC_Reset();

    /* ADC Common Configuration */
    adcCommonConfig.mode = ADC_MODE_TRIPLE_INTERL;
    adcCommonConfig.prescaler = ADC_PRESCALER_DIV4;
    adcCommonConfig.accessMode = ADC_ACCESS_MODE_2;
    adcCommonConfig.twoSampling = ADC_TWO_SAMPLING_5CYCLES;
    ADC_CommonConfig(&adcCommonConfig);

    /* ADC Configuration */
    ADC_ConfigStructInit(&adcConfig);

    /* Set resolution*/
    adcConfig.resolution = ADC_RESOLUTION_12BIT;
    /* Set dataAlign*/
    adcConfig.dataAlign = ADC_DATA_ALIGN_RIGHT;
    /* Set scanDir*/
    adcConfig.scanConvMode = DISABLE;
    /* Set convMode continous*/
    adcConfig.continuousConvMode = ENABLE;
    /* Set extTrigEdge*/
    adcConfig.extTrigEdge = ADC_EXT_TRIG_EDGE_NONE;

    ADC_Config(ADC1, &adcConfig);
    ADC_ConfigRegularChannel(ADC1, ADC_CHANNEL_0, 1, ADC_SAMPLETIME_3CYCLES);

    ADC_Config(ADC2, &adcConfig);
    ADC_ConfigRegularChannel(ADC2, ADC_CHANNEL_0, 1, ADC_SAMPLETIME_3CYCLES);

    ADC_Config(ADC3, &adcConfig);
    ADC_ConfigRegularChannel(ADC3, ADC_CHANNEL_0, 1, ADC_SAMPLETIME_3CYCLES);

    ADC_EnableMultiModeDMARequest();

    /* Enable ADC*/
    ADC_Enable(ADC1);
    ADC_Enable(ADC2);
    ADC_Enable(ADC3);
    ADC_SoftwareStartConv(ADC1);
}

/*!
 * @brief     DMA Init
 *
 * @param     None
 *
 * @retval    None
 */
void DMA_Init(uint32_t* Buf)
{
    /* DMA Configure */
    DMA_Config_T dmaConfig;

    /* Enable DMA clock */
    RCM_EnableAHB1PeriphClock(RCM_AHB1_PERIPH_DMA2);

    /* size of buffer*/
    dmaConfig.bufferSize = 3;
    /* set memory Data Size*/
    dmaConfig.memoryDataSize = DMA_MEMORY_DATA_SIZE_WORD;
    /* Set peripheral Data Size*/
    dmaConfig.peripheralDataSize = DMA_PERIPHERAL_DATA_SIZE_WORD;
    /* Enable Memory Address increase*/
    dmaConfig.memoryInc = DMA_MEMORY_INC_ENABLE;
    /* Disable Peripheral Address increase*/
    dmaConfig.peripheralInc = DMA_PERIPHERAL_INC_DISABLE;
    /* Reset Circular Mode*/
    dmaConfig.loopMode = DMA_MODE_CIRCULAR;
    /* set priority*/
    dmaConfig.priority = DMA_PRIORITY_HIGH;
    /* read from peripheral*/
    dmaConfig.dir = DMA_DIR_PERIPHERALTOMEMORY;
    /* Set memory Address*/
    dmaConfig.memoryBaseAddr = (uint32_t)Buf;
    /* Set Peripheral Address*/
    dmaConfig.peripheralBaseAddr = (uint32_t)&ADC->CDATA;

    dmaConfig.channel = DMA_CHANNEL_0;
    dmaConfig.fifoMode = DMA_FIFOMODE_DISABLE;
    dmaConfig.fifoThreshold = DMA_FIFOTHRESHOLD_HALFFULL;
    dmaConfig.peripheralBurst = DMA_PERIPHERALBURST_SINGLE;
    dmaConfig.memoryBurst = DMA_MEMORYBURST_SINGLE;

    DMA_Config(DMA2_Stream0, &dmaConfig);

    /* Clear DMA TF flag*/
    DMA_ClearIntFlag(DMA2_Stream0, DMA_INT_TCIFLG0);

    /* Enable DMA Interrupt*/
    DMA_EnableInterrupt(DMA2_Stream0, DMA_INT_TCIFLG);
    DMA_Enable(DMA2_Stream0);
}

#if defined (__CC_ARM) || defined (__ICCARM__) || (defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050))

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       ch:  The characters that need to be send.
*
* @param       *f:  pointer to a FILE that can recording all information
*              needed to control a stream
*
* @retval      The characters that need to be send.
*
* @note
*/
int fputc(int ch, FILE* f)
{
    /* send a byte of data to the serial port */
    USART_TxData(DEBUG_USART, (uint8_t)ch);

    /* wait for the data to be send  */
    while (USART_ReadStatusFlag(DEBUG_USART, USART_FLAG_TXBE) == RESET);

    return (ch);
}

#elif defined (__GNUC__)

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       ch:  The characters that need to be send.
*
* @retval      The characters that need to be send.
*
* @note
*/
int __io_putchar(int ch)
{
    /* send a byte of data to the serial port */
    USART_TxData(DEBUG_USART, ch);

    /* wait for the data to be send  */
    while (USART_ReadStatusFlag(DEBUG_USART, USART_FLAG_TXBE) == RESET);

    return ch;
}

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       file:  Meaningless in this function.
*
* @param       *ptr:  Buffer pointer for data to be sent.
*
* @param       len:  Length of data to be sent.
*
* @retval      The characters that need to be send.
*
* @note
*/
int _write(int file, char* ptr, int len)
{
    int i;

    UNUSED(file);

    for (i = 0; i < len; i++)
    {
        __io_putchar(*ptr++);
    }

    return len;
}

#else
#warning Not supported compiler type
#endif
