/**
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2025-01-01
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"
#include <stdio.h>

/* Private includes *******************************************************/
#include "g32m3101_device_cfg.h"

/* Private macro **********************************************************/
#define ADC_STAB_DELAY_US               3U

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
__IO uint16_t adc1ConvValue[2] = {0U};

/* Private function prototypes ********************************************/

/* External variables *****************************************************/
uint8_t conv_flag = 0U;

/* External functions *****************************************************/

/**
 * @brief   Main program
 *
 * @param   None
 *
 * @retval  None
 */
int main(void)
{
    uint32_t counter;
    /* Device configuration */
    DDL_DeviceConfig();

    /* Enable the Peripheral */
    DDL_ADC_Enable(ADC1);

    /* Delay for ADC stabilization time */
    /* Compute number of CPU cycles to wait for */
    counter = (ADC_STAB_DELAY_US * (SystemCoreClock / 1000000U));
    while(counter != 0U)
    {
      counter--;
    }

    DDL_ADC_ClearFlag_EOC(ADC1);
    DDL_ADC_ClearFlag_OVR(ADC1);

    DDL_DMA_EnableChannel(DMA1, DDL_DMA_CHANNEL_1);

    DDL_ADC_StartConversion(ADC1);
    counter = (ADC_STAB_DELAY_US * (SystemCoreClock / 1000000U));
    while(counter != 0U)
    {
      counter--;
    }

    /* Infinite loop */
    while (1)
    {
        if(conv_flag == 1)
        {
            printf("ADC1 CH4(PB1) Value: %d mV\r\n", (adc1ConvValue[0] * 5000U) / 4095U);
            printf("ADC1 CH5(PB2) Value: %d mV\r\n", (adc1ConvValue[1] * 5000U) / 4095U);

            DDL_ADC_StartConversion(ADC1);
        }
        else if(conv_flag == 2)
        {
            conv_flag = 0;
            printf("ADC1 CH2(PB5) Value: %d mV\r\n", (adc1ConvValue[0] * 5000U) / 4095U);
            printf("ADC1 CH3(PB6) Value: %d mV\r\n", (adc1ConvValue[1] * 5000U) / 4095U);

            DDL_ADC_StartConversion(ADC1);
        }
        /* Toggle LED2 every 500 ms */
        BOARD_LED_Toggle(LED2);
        DDL_mDelay(500U);
    }
}

void DDL_DMA_IRQHandler(void)
{
    if(DDL_DMA_IsActiveFlag_TC1(DMA1))
    {
        DDL_DMA_ClearFlag_TC1(DMA1);
        conv_flag++;
    }
}


#if defined (__CC_ARM) || defined (__ICCARM__) || (defined(__ARMCC_VERSION) && (__ARMCC_VERSION >= 6010050))

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       ch:  The characters that need to be send.
*
* @param       *f:  pointer to a FILE that can recording all information
*              needed to control a stream
*
* @retval      The characters that need to be send.
*
* @note
*/
int fputc(int ch, FILE* f)
{
    /* send a byte of data to the serial port */
    DDL_USART_TransmitData8(UART0, (uint8_t)ch);
    while(DDL_USART_IsActiveFlag_TXE(UART0) == RESET);

    return (ch);
}

#elif defined (__GNUC__)

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       ch:  The characters that need to be send.
*
* @retval      The characters that need to be send.
*
* @note
*/
int __io_putchar(int ch)
{
    /* send a byte of data to the serial port */
    DDL_USART_TransmitData8(UART0, (uint8_t)ch);
    while(DDL_USART_IsActiveFlag_TXE(UART0) == RESET);

    return ch;
}

/*!
* @brief       Redirect C Library function printf to serial port.
*              After Redirection, you can use printf function.
*
* @param       file:  Meaningless in this function.
*
* @param       *ptr:  Buffer pointer for data to be sent.
*
* @param       len:  Length of data to be sent.
*
* @retval      The characters that need to be send.
*
* @note
*/
int _write(int file, char* ptr, int len)
{
    int i;
    for (i = 0; i < len; i++)
    {
        __io_putchar(*ptr++);
    }

    return len;
}

#else
#warning Not supported compiler type
#endif


