/**
 * @file        g32m3101_timebase_cfg.c
 *
 * @brief       This file provides configuration support for timebase
 *
 * @version     V1.0.0
 *
 * @date        2025-06-01
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "g32m3101_ddl_atmr.h"
#include "g32m3101_ddl_scu.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/
DDL_ATMR_InitTypeDef ATMR_InitStruct;
DDL_SCU_ClocksTypeDef SYSCLOCK_InitStruct;
__IO uint32_t uwTick;

/* Private function prototypes ********************************************/

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief Redefine the function DDL_Init1msTick.
 *        This function configures 1ms delay with time base, rather than systick.
 *
 * @param HCLKFrequency HCLK frequency in Hz
 *
 * @retval None
 */
void DDL_Init1msTick(uint32_t HCLKFrequency)
{
    uint32_t tmrClock;
    uint32_t prescalerValue = 0U;

    /* Enable TMR0 clock */
    DDL_SCU_Unlock();
    DDL_SCU_EnableAPBPeripheralClock(DDL_SCU_APB_PERIPHERAL_ATMR);
    DDL_SCU_Lock();

    DDL_SCU_GetSysctrlClocksFreq(&SYSCLOCK_InitStruct);
    tmrClock = SYSCLOCK_InitStruct.PCLK_Frequency;

    /* Compute the prescaler value to have TMR0 counter clock equal to 1MHz */
    prescalerValue = (uint32_t) ((tmrClock / 1000000U) - 1U);

    /* Initialize TMR0 */
    /* Initialize TMR0 peripheral as follow:
    Period = [(TMR0CLK/1000) - 1]. to have a (1/1000) s time base
    Prescaler = (tmrClock / 1000000 - 1) to have a 1MHz counter clock
    */
    ATMR_InitStruct.Prescaler            = prescalerValue;
    ATMR_InitStruct.CounterMode          = DDL_ATMR_COUNTERMODE_UP;
    ATMR_InitStruct.Autoreload           = (1000000U / 1000U) - 1U;
    ATMR_InitStruct.ClockDivision        = DDL_ATMR_CLOCKDIVISION_DIV1;
    ATMR_InitStruct.RepetitionCounter    = 0U;

    DDL_ATMR_DisableARRPreload(ATMR);
    DDL_ATMR_Init(ATMR, &ATMR_InitStruct);
    if(DDL_ATMR_IsActiveFlag_UPDATE(ATMR))
    {
        DDL_ATMR_ClearFlag_UPDATE(ATMR);
    }

    DDL_ATMR_EnableIT_UPDATE(ATMR);

    DDL_ATMR_EnableCounter(ATMR);

    /* Enable and set ATMR Interrupt priority */
    NVIC_SetPriority((IRQn_Type)(ATMR_BRK_UP_TRG_COM_IRQn), NVIC_EncodePriority(NVIC_GetPriorityGrouping(), 0xFu, 0u));
    NVIC_EnableIRQ((IRQn_Type)(ATMR_BRK_UP_TRG_COM_IRQn));
}

/**
 * @brief  Suspend Tick increment
 *
 * @note   Disable the tick increment by disabling TMR update interrupt
 *
 * @retval None
 */
void DDL_SuspendTick(void)
{
    /* Disable TMR0 update Interrupt */
    DDL_ATMR_DisableIT_UPDATE(ATMR);
}

/**
  * @brief  Resume Tick increment.
  * @note   Enable the tick increment by Enabling TMR update interrupt.
  * @retval None
  */
void DDL_ResumeTick(void)
{
    /* Enable TMR0 Update interrupt */
    DDL_ATMR_EnableIT_UPDATE(ATMR);
}

/**
  * @brief  This function provides accurate delay (in milliseconds)
  * @param  Delay specifies the delay time length, in milliseconds.
  * @retval None
  */
void DDL_mDelay(uint32_t Delay)
{
  uint32_t tickstart = uwTick;
  uint32_t wait = Delay;

  /* Add a freq to guarantee minimum wait */
  if (wait < 0xFFFFFFFFU)
  {
    wait ++;
  }

  while((uwTick - tickstart) < wait)
  {
  }
}

/**
 * @brief  This function handles TMR interrupt request.
 * @retval None
 */
void ATMR_BRK_UP_TRG_COM_IRQHandler(void)
{
    DDL_ATMR_ClearFlag_COM(ATMR);
    DDL_ATMR_ClearFlag_TRIG(ATMR);
    DDL_ATMR_ClearFlag_UPDATE(ATMR);
    uwTick++;
}
