/**
 *
 * @file        main.c
 *
 * @brief       Main program body
 *
 * @version     V1.0.0
 *
 * @date        2025-10-30
 *
 * @attention
 *
 *  Copyright (C) 2025 Geehy Semiconductor
 *
 *  You may not use this file except in compliance with the
 *  GEEHY COPYRIGHT NOTICE (GEEHY SOFTWARE PACKAGE LICENSE).
 *
 *  The program is only for reference, which is distributed in the hope
 *  that it will be useful and instructional for customers to develop
 *  their software. Unless required by applicable law or agreed to in
 *  writing, the program is distributed on an "AS IS" BASIS, WITHOUT
 *  ANY WARRANTY OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the GEEHY SOFTWARE PACKAGE LICENSE for the governing permissions
 *  and limitations under the License.
 */

/* Includes ***************************************************************/
#include "main.h"

/* Private includes *******************************************************/

/* Private macro **********************************************************/

/* Private typedef ********************************************************/

/* Private variables ******************************************************/

/* Private function prototypes ********************************************/
void DDL_SysClkConfig(void);
void TMR_Init(void);

/* External variables *****************************************************/

/* External functions *****************************************************/

/**
 * @brief     Main program
 *
 * @param     None
 *
 * @retval    None
 *
 */
int main(void)
{
    /* Configure system clock */
    DDL_SysClkConfig();

    /* Initialize TMR */
    TMR_Init();

    while (1)
    {
    }
}

/**
 * @brief   System clock configuration
 *
 * @param   None
 *
 * @retval  None
 */
void DDL_SysClkConfig(void)
{
    /* Unlock clock control registers */
    /* Wait until the registers are unlocked */
    DDL_RCM_Unlock();
    while ((RCM->KEY & RCM_KEY_KEYST) != RCM_KEY_KEYST) 
    { 
    }

    /* Enable HSE and wait for ready */
    /* Enable HSE input clock */
    DDL_RCM_HSE_Enable();
    /* Wait until HSE is ready */
    while (DDL_RCM_HSE_IsReady() != 1) 
    {
        /* wait for HSERDY */
    }

    /* Configure Flash wait states appropriate for 120 MHz */
    /* Set Flash wait period to accommodate higher frequency */
    DDL_FLASH_SetWaitPeriod(FLASH_DDL_WAIT_PERIOD_3);
    /* Set Flash erase time base to 120 MHz (use 119 to represent 120 MHz) */
    DDL_FLASH_SetEraseTimeBase(119);

    /* Configure PLL */
    /* Ensure PLL is disabled prior to configuration */
    DDL_RCM_PLL_Disable();

    /* Configure PLL settings: multiplier, prescaler, and clock source */
    /* PLL multiplier set to 15 (x15), prescaler no division, clock source from HSE */
    DDL_RCM_PLL_SetMultiplier(15);
    DDL_RCM_PLL_SetPrescaler(DDL_RCM_PLL_DIV1);
    DDL_RCM_PLL_SetClkSource(DDL_RCM_PLL_CLKSOURCE_HSE);

    /* Enable PLL and wait for ready */
    DDL_RCM_PLL_Enable();
    while (DDL_RCM_PLL_IsReady() != 1)
    {
        /* wait for PLL Ready */
    }

    /* Switch system clock to PLL output */
    DDL_RCM_SetSysClkSource(DDL_RCM_SYS_CLKSOURCE_PLL);

    /* Enable clock switch and wait for completion */
    DDL_RCM_EnableSysClkSwitch();
    while (DDL_RCM_IsActiveFlag_SWDONE() != 1)
    {
    }

    /* Set AHB, APB prescalers */
    DDL_RCM_SetAHBPrescaler(DDL_RCM_AHB_DIV_1);
    DDL_RCM_SetAPBPrescaler(DDL_RCM_APB_DIV_1);
    
    /* Disable clock switch controls */
    DDL_RCM_DisableSysClkSwitch();
    
    /* Lock clock control registers */
    DDL_RCM_Unlock();

    /* Update SystemCoreClock if used by the project */
    SystemCoreClockUpdate();
}

/**
 * @brief     TMR Init
 *
 * @param     None
 *
 * @retval    None
 *
 */
void TMR_Init(void)
{
    DDL_GPIO_InitTypeDef GPIO_InitStruct = {0U};
    DDL_TMR_InitTypeDef TMR_InitStruct = {0U};
    DDL_TMR_OC_InitTypeDef TMR_OC_InitStruct = {0U};

    /* Config Clock */
    DDL_RCM_Unlock();
    DDL_RCM_EnableAPBPeripheral(DDL_RCM_APB_PERIPHERAL_TMR2);
    DDL_RCM_EnableAPBPeripheral(DDL_RCM_APB_PERIPHERAL_TMR3);
    DDL_RCM_EnableAPBPeripheral(DDL_RCM_APB_PERIPHERAL_TMR4);
    DDL_RCM_EnableAHBPeripheral(DDL_RCM_AHB_PERIPHERAL_GPIO);
    DDL_RCM_Lock();

    /* Configure TMR2 channel 1 GPIO PC4 */
    GPIO_InitStruct.Pin        = DDL_GPIO_PIN_4;
    GPIO_InitStruct.Mode       = DDL_GPIO_MODE_ALTERNATE;
    GPIO_InitStruct.Speed      = DDL_GPIO_SPEED_FREQ_HIGH;
    GPIO_InitStruct.OutputType = DDL_GPIO_OUTPUT_PUSHPULL;
    GPIO_InitStruct.Pull       = DDL_GPIO_PULL_NO;
    GPIO_InitStruct.Alternate  = DDL_GPIO_AF_0;
    DDL_GPIO_Init(GPIOC, &GPIO_InitStruct);

    /* Configure TMR3 channel 1 GPIO PB2 */
    GPIO_InitStruct.Pin        = DDL_GPIO_PIN_2;
    GPIO_InitStruct.Alternate  = DDL_GPIO_AF_2;
    DDL_GPIO_Init(GPIOB, &GPIO_InitStruct);

    /* Configure TMR4 channel 1 GPIO PC6 */
    GPIO_InitStruct.Pin        = DDL_GPIO_PIN_6;
    GPIO_InitStruct.Alternate  = DDL_GPIO_AF_0;
    DDL_GPIO_Init(GPIOC, &GPIO_InitStruct);

    /* Configure TMR2, TMR3, TMR4 */
    /* TMR2 Configuration: */
    /* Prescaler = 0 (no division, clock frequency = 120 MHz) */
    /* Autoreload = 95 */
    /* TMR2 update event frequency = main clock / (Autoreload + 1) */
    /* TMR2 update event frequency = 120 MHz / (95 + 1) = 1.25 MHz */
    /* TMR2 PWM frequency = 1.25 MHz */

    TMR_InitStruct.Prescaler         = 0;
    TMR_InitStruct.CounterMode       = DDL_TMR_COUNTERMODE_UP;
    TMR_InitStruct.Autoreload        = 95;
    TMR_InitStruct.ClockDivision     = DDL_TMR_CLOCKDIVISION_DIV1;
    TMR_InitStruct.RepetitionCounter = 0x00000000U;
    DDL_TMR_Init(TMR2, &TMR_InitStruct);

    /* TMR3 Configuration: */
    /* TMR3 is triggered by TMR2's update event (frequency = 1.25 MHz) */
    /* Prescaler = 0 (no division, clock frequency = 1.25 MHz) */
    /* Autoreload = 3 */
    /* TMR3 PWM frequency = TMR2 update event frequency / (Autoreload + 1) */
    /* TMR3 PWM frequency = 1.25 MHz / (3 + 1) = 312.5 kHz */
    TMR_InitStruct.Autoreload        = 3;
    DDL_TMR_Init(TMR3, &TMR_InitStruct);

    /* TMR4 Configuration: */
    /* TMR4 is triggered by TMR2's update event (frequency = 1.25 MHz) */
    /* Prescaler = 0 (no division, clock frequency = 1.25 MHz) */
    /* Autoreload = 1 */
    /* TMR4 PWM frequency = TMR2 update event frequency / (Autoreload + 1) */
    /* TMR4 PWM frequency = 1.25 MHz / (1 + 1) = 625 kHz */
    TMR_InitStruct.Autoreload        = 1;
    DDL_TMR_Init(TMR4, &TMR_InitStruct);

    DDL_TMR_OC_StructInit(&TMR_OC_InitStruct);

    /* Configure TMR2, TMR3, TMR4 channel 1 */
    TMR_OC_InitStruct.OCMode       = DDL_TMR_OCMODE_PWM1;
    TMR_OC_InitStruct.OCState      = DDL_TMR_OCSTATE_ENABLE;
    TMR_OC_InitStruct.CompareValue = 48;
    TMR_OC_InitStruct.OCPolarity   = DDL_TMR_OCPOLARITY_HIGH;
    DDL_TMR_OC_Init(TMR2, DDL_TMR_CHANNEL_CH1, &TMR_OC_InitStruct);

    TMR_OC_InitStruct.CompareValue = 2;
    DDL_TMR_OC_Init(TMR3, DDL_TMR_CHANNEL_CH1, &TMR_OC_InitStruct);

    TMR_OC_InitStruct.CompareValue = 1;
    DDL_TMR_OC_Init(TMR4, DDL_TMR_CHANNEL_CH1, &TMR_OC_InitStruct);

    /* Enable Master-Slave Mode for TMR2 */
    DDL_TMR_EnableMasterSlaveMode(TMR2);

    /* Set TMR2 Trigger Output to Update Event */
    /* TMR2 will generate a trigger output (TRGO) signal on each update event. */
    DDL_TMR_SetTriggerOutput(TMR2, DDL_TMR_TRGO_UPDATE);

    /* Configure TMR3 as a Slave Timer */
    /* TMR3 is triggered by TMR2's update event (TRGO signal). */
    DDL_TMR_SetTriggerInput(TMR3, DDL_TMR_TS_ITR1);

    /* Set TMR3 to Gated Slave Mode */
    DDL_TMR_SetSlaveMode(TMR3, DDL_TMR_SLAVEMODE_GATED);

    /* Configure TMR4 as a Slave Timer */
    /* TMR4 is also triggered by TMR2's update event (TRGO signal). */
    DDL_TMR_SetTriggerInput(TMR4, DDL_TMR_TS_ITR1);

    /* Set TMR4 to Gated Slave Mode */
    DDL_TMR_SetSlaveMode(TMR4, DDL_TMR_SLAVEMODE_GATED);

    /* Enable TMRx */
    DDL_TMR_EnableCounter(TMR2);
    DDL_TMR_EnableCounter(TMR3);
    DDL_TMR_EnableCounter(TMR4);
}
